// ========================================
// GESTION DES GRAPHIQUES - VERSION TV RÉTRO AMÉLIORÉE
// ========================================

let mainChart = null;
let currentChartIndex = 0;
let currentYearIndex = 0;
let currentMediaFilter = 'all'; // 'all', 'tv', 'radio'
let years = [];
let artisticOverlay = null;

// Palette de couleurs TV Rétro
const TV_COLORS = {
    female: '#ff6b8b',
    male: '#4a9eff',
    radio: '#50c878',
    text: '#ffffff',
    grid: 'rgba(0, 0, 0, 0.1)',
    accent: '#0066aa',
    background: 'rgba(255, 255, 255, 0.3)'
};

// Liste des chaînes TV (34 chaînes)
const FULL_TV_LIST = ["Arte", "Animaux", "BFM TV", "Canal+", "Canal+ Sport", "Chasse et Pêche", "Chérie 25", "Comédie+", "D8/C8", "Euronews", "Eurosport France", "France 2", "France 24", "France 3", "France 5", "France Ô", "Histoire", "I-Télé/CNews", "L'Équipe 21", "LCI", "LCP/Public Sénat", "La Chaîne Météo", "M6", "Monte Carlo TMC", "NRJ 12", "Paris Première", "Planète+", "TF1", "TV Breizh", "TV5 Monde", "Toute l'Histoire", "Téva", "Voyage", "W9"];

// Liste des chaînes Radio (21 chaînes)
const FULL_RADIO_LIST = ["Chérie FM", "Europe 1", "France Bleu", "France Culture", "France Info", "France Inter", "France Musique", "Fun Radio", "Mouv'", "NRJ", "Nostalgie", "RFM", "RMC", "RTL", "RTL 2", "Radio Classique", "Radio France Internationale", "Rire et Chansons", "Skyrock", "Sud Radio", "Virgin Radio"];

let selectedChannels = [];

// Données de durée totale des programmes par chaîne (en heures)
// Ces données devraient être remplacées par vos données réelles
const PROGRAM_DURATION_DATA = {
    '2019': {
        'TF1': 1850,
        'France 2': 2100,
        'France 3': 1950,
        'M6': 1750,
        'ARTE': 1200,
        'Canal+': 1600,
        'BFM TV': 2200,
        'France 24': 1900,
        'Arte': 1200,
        'Animaux': 800,
        'BFM TV': 2200,
        'Canal+ Sport': 1400,
        'Chasse et Pêche': 600,
        'Chérie 25': 900,
        'Comédie+': 1100,
        'D8/C8': 1300,
        'Euronews': 1800,
        'Eurosport France': 1500,
        'France 5': 1700,
        'France Ô': 950,
        'Histoire': 850,
        'I-Télé/CNews': 2000,
        'L\'Équipe 21': 1250,
        'LCI': 1950,
        'LCP/Public Sénat': 1150,
        'La Chaîne Météo': 1650,
        'Monte Carlo TMC': 1350,
        'NRJ 12': 1450,
        'Paris Première': 1050,
        'Planète+': 950,
        'TV Breizh': 750,
        'TV5 Monde': 1850,
        'Toute l\'Histoire': 850,
        'Téva': 1000,
        'Voyage': 900,
        'W9': 1550
    }
    // Ajoutez d'autres années si disponibles
};

// ===== GRAPHIQUE D'ÉVOLUTION TEMPORELLE EN D3.JS =====
function renderEvolutionChartD3(containerId, config) {
    const container = document.getElementById(containerId);
    if (!container) return null;
    
    // Effacer le contenu précédent
    container.innerHTML = '';
    
    // Dimensions
    const margin = { top: 40, right: 120, bottom: 50, left: 60 };
    const width = container.clientWidth - margin.left - margin.right;
    const height = 400 - margin.top - margin.bottom;
    
    // Récupérer les données
    const data = config.getData();
    if (!data || !data.labels || !data.datasets) {
        container.innerHTML = '<div class="no-data">Aucune donnée disponible</div>';
        return;
    }
    
    // Années (labels du graphique d'évolution)
    const yearsData = data.labels;
    const datasets = data.datasets;
    
    if (datasets.length === 0) {
        container.innerHTML = '<div class="no-data">Aucune chaîne sélectionnée</div>';
        return;
    }
    
    // Créer le SVG
    const svg = d3.select(`#${containerId}`)
        .append('svg')
        .attr('width', width + margin.left + margin.right)
        .attr('height', height + margin.top + margin.bottom)
        .append('g')
        .attr('transform', `translate(${margin.left}, ${margin.top})`);
    
    // Échelles
    const xScale = d3.scalePoint()
        .domain(yearsData)
        .range([0, width])
        .padding(0.5);
    
    const yScale = d3.scaleLinear()
        .domain([0, 100])
        .range([height, 0])
        .nice();
    
    // Générateur de ligne
    const lineGenerator = d3.line()
        .x(d => xScale(d.year))
        .y(d => yScale(d.value))
        .curve(d3.curveMonotoneX);
    
    // Préparer les données pour chaque chaîne
    const lineData = datasets.map((dataset, index) => {
        const dataPoints = yearsData.map((year, i) => ({
            year: year,
            value: dataset.data[i] || 0,
            label: dataset.label
        }));
        
        return {
            label: dataset.label,
            data: dataPoints,
            color: dataset.borderColor || getColorByIndex(index),
            lastPoint: dataPoints[dataPoints.length - 1]
        };
    });
    
    // Fonction utilitaire pour obtenir une couleur par index
    function getColorByIndex(index) {
        const colors = [
            '#ff6b8b', // Rose
            '#4a9eff', // Bleu
            '#50c878', // Vert
            '#ff9f43', // Orange
            '#a370f7', // Violet
            '#00d4ff', // Cyan
            '#ff4757', // Rouge vif
            '#2ed573'  // Vert vif
        ];
        return colors[index % colors.length];
    }
    
    // Axe X
    svg.append('g')
        .attr('class', 'x-axis')
        .attr('transform', `translate(0, ${height})`)
        .call(d3.axisBottom(xScale)
            .tickFormat(d => d)
            .tickSizeOuter(0))
        .selectAll('text')
        .style('fill', TV_COLORS.text)
        .style('font-family', 'Segoe UI')
        .style('font-size', '10px')
        .attr('dy', '1em');
    
    // Titre axe X
    svg.append('text')
        .attr('class', 'axis-title')
        .attr('x', width / 2)
        .attr('y', height + margin.bottom - 10)
        .attr('text-anchor', 'middle')
        .style('fill', TV_COLORS.text)
        .style('font-family', 'Segoe UI')
        .style('font-size', '12px')
        .style('font-weight', '600')
        .text('Années');
    
    // Axe Y
    svg.append('g')
        .attr('class', 'y-axis')
        .call(d3.axisLeft(yScale)
            .tickFormat(d => d + '%')
            .ticks(6)
            .tickSizeOuter(0))
        .selectAll('text')
        .style('fill', TV_COLORS.text)
        .style('font-family', 'Segoe UI')
        .style('font-size', '10px');
    
    // Titre axe Y
    svg.append('text')
        .attr('class', 'axis-title')
        .attr('transform', 'rotate(-90)')
        .attr('x', -height / 2)
        .attr('y', -margin.left + 15)
        .attr('text-anchor', 'middle')
        .style('fill', TV_COLORS.text)
        .style('font-family', 'Segoe UI')
        .style('font-size', '12px')
        .style('font-weight', '600')
        .text('Taux de parole des femmes (%)');
    
    // Grille horizontale
    svg.append('g')
        .attr('class', 'grid')
        .call(d3.axisLeft(yScale)
            .tickSize(-width)
            .tickFormat('')
            .ticks(6))
        .selectAll('line')
        .style('stroke', TV_COLORS.grid)
        .style('stroke-width', 0.5)
        .style('stroke-dasharray', '3,3');
    
    // Dessiner les lignes
    const lines = svg.selectAll('.line-group')
        .data(lineData)
        .enter()
        .append('g')
        .attr('class', 'line-group');
    
    lines.append('path')
        .attr('class', 'line')
        .attr('d', d => lineGenerator(d.data))
        .style('fill', 'none')
        .style('stroke', d => d.color)
        .style('stroke-width', 3)
        .style('stroke-linecap', 'round')
        .style('stroke-linejoin', 'round');
    
    // Points sur les lignes
    lines.selectAll('.point')
        .data(d => d.data)
        .enter()
        .append('circle')
        .attr('class', 'point')
        .attr('cx', d => xScale(d.year))
        .attr('cy', d => yScale(d.value))
        .attr('r', 4)
        .style('fill', (d, i, nodes) => {
            const parentData = d3.select(nodes[i].parentNode).datum();
            return parentData.color;
        })
        .style('stroke', '#ffffff')
        .style('stroke-width', 2)
        .style('cursor', 'pointer')
        .on('mouseover', function(event, d) {
            // Afficher le tooltip
            showTooltip(event, d);
            // Agrandir le point
            d3.select(this)
                .transition()
                .duration(200)
                .attr('r', 7);
        })
        .on('mouseout', function(event, d) {
            // Cacher le tooltip
            hideTooltip();
            // Réduire le point
            d3.select(this)
                .transition()
                .duration(200)
                .attr('r', 4);
        });
    
    // Labels de fin de ligne
    const endLabels = lines.append('g')
        .attr('class', 'end-label')
        .attr('transform', d => {
            const lastPoint = d.data[d.data.length - 1];
            return `translate(${xScale(lastPoint.year)}, ${yScale(lastPoint.value)})`;
        });
    
    endLabels.append('circle')
        .attr('r', 6)
        .style('fill', d => d.color)
        .style('stroke', '#ffffff')
        .style('stroke-width', 2);
    
    endLabels.append('text')
        .attr('dx', 12)
        .attr('dy', 4)
        .text(d => d.label)
        .style('fill', d => d.color)
        .style('font-family', 'Segoe UI')
        .style('font-size', '11px')
        .style('font-weight', '500');
    
    // Légende interactive
    const legend = svg.append('g')
        .attr('class', 'legend')
        .attr('transform', `translate(${width + 20}, 0)`);
    
    const legendItems = legend.selectAll('.legend-item')
        .data(lineData)
        .enter()
        .append('g')
        .attr('class', 'legend-item')
        .attr('transform', (d, i) => `translate(0, ${i * 25})`)
        .style('cursor', 'pointer')
        .on('click', function(event, d) {
            toggleD3Line(d.label);
        });
    
    legendItems.append('rect')
        .attr('width', 15)
        .attr('height', 15)
        .attr('rx', 3)
        .style('fill', d => d.color)
        .style('stroke', '#ffffff')
        .style('stroke-width', 1);
    
    legendItems.append('text')
        .attr('x', 22)
        .attr('y', 12)
        .text(d => d.label)
        .style('fill', TV_COLORS.text)
        .style('font-family', 'Segoe UI')
        .style('font-size', '11px')
        .style('user-select', 'none');
    
    // Titre du graphique
    const chartTitle = getChartTitle(config, currentMediaFilter);
    svg.append('text')
        .attr('class', 'chart-title')
        .attr('x', width / 2)
        .attr('y', -10)
        .attr('text-anchor', 'middle')
        .style('fill', TV_COLORS.text)
        .style('font-family', 'Segoe UI')
        .style('font-size', '16px')
        .style('font-weight', '600')
        .text(chartTitle);
    
    // Ajouter une description d'analyse sous le titre
    const descriptionText = getChartDescription(config, lineData);
    svg.append('text')
        .attr('class', 'chart-description')
        .attr('x', width / 2)
        .attr('y', 15)
        .attr('text-anchor', 'middle')
        .style('fill', TV_COLORS.text)
        .style('font-family', 'Segoe UI')
        .style('font-size', '12px')
        .style('font-weight', '400')
        .style('opacity', 0.8)
        .text(descriptionText);
    
    // Fonction pour afficher/masquer les lignes D3
    function toggleD3Line(channelName) {
        const lineGroup = svg.selectAll('.line-group')
            .filter(d => d.label === channelName);
        
        const isVisible = lineGroup.style('opacity') !== '0';
        
        lineGroup.transition()
            .duration(300)
            .style('opacity', isVisible ? 0 : 1);
        
        // Mettre à jour la légende
        const legendItem = legend.selectAll('.legend-item')
            .filter(d => d.label === channelName);
        
        legendItem.select('rect')
            .transition()
            .duration(300)
            .style('opacity', isVisible ? 0.3 : 1);
        
        legendItem.select('text')
            .transition()
            .duration(300)
            .style('opacity', isVisible ? 0.3 : 1)
            .style('text-decoration', isVisible ? 'line-through' : 'none');
        
        // Effet glitch
        triggerGlitchEffect(150);
    }
    
    // Fonctions pour le tooltip D3
    function showTooltip(event, dataPoint) {
        const tooltip = d3.select('#d3-tooltip');
        
        if (tooltip.empty()) {
            d3.select('body')
                .append('div')
                .attr('id', 'd3-tooltip')
                .style('position', 'absolute')
                .style('background', 'rgba(26, 26, 26, 0.95)')
                .style('color', '#ffffff')
                .style('padding', '12px')
                .style('border-radius', '6px')
                .style('border', `1px solid ${TV_COLORS.accent}`)
                .style('font-family', 'Segoe UI')
                .style('font-size', '11px')
                .style('pointer-events', 'none')
                .style('z-index', '1000')
                .style('min-width', '150px');
        }
        
        const tooltipDiv = d3.select('#d3-tooltip');
        
        tooltipDiv.html(`
            <div style="margin-bottom: 5px; font-weight: 600; font-size: 12px;">${dataPoint.label}</div>
            <div style="margin-bottom: 3px;"><strong>Année:</strong> ${dataPoint.year}</div>
            <div style="margin-bottom: 3px;"><strong>Taux femmes:</strong> ${dataPoint.value.toFixed(1)}%</div>
            <div><strong>Taux hommes:</strong> ${(100 - dataPoint.value).toFixed(1)}%</div>
        `);
        
        tooltipDiv.style('left', (event.pageX + 15) + 'px')
                 .style('top', (event.pageY - 15) + 'px')
                 .style('opacity', 1);
    }
    
    function hideTooltip() {
        d3.select('#d3-tooltip')
            .transition()
            .duration(200)
            .style('opacity', 0);
    }
    
    // Retourner le nœud SVG
    return svg.node();
}

// ===== GRAPHIQUE À BULLES EN D3.JS (VERSION AMÉLIORÉE) =====
function renderBubbleChartD3(containerId, config) {
    const container = document.getElementById(containerId);
    if (!container) return null;
    
    // Effacer le contenu précédent
    container.innerHTML = '';
    
    // Dimensions ajustées pour mieux afficher les labels
    const margin = { top: 50, right: 120, bottom: 50, left: 80 }; // Réduit top et bottom
    const width = container.clientWidth - margin.left - margin.right;
    const height = 400 - margin.top - margin.bottom; // Réduit à 400px
    
    // Récupérer les données
    const year = document.getElementById('yearFilter')?.value || '2019';
    const data = config.getData(year);
    
    if (!data || !data.datasets || data.datasets.length === 0) {
        container.innerHTML = '<div class="no-data">Aucune donnée disponible</div>';
        return;
    }
    
    // Créer le SVG
    const svg = d3.select(`#${containerId}`)
        .append('svg')
        .attr('width', width + margin.left + margin.right)
        .attr('height', height + margin.top + margin.bottom)
        .append('g')
        .attr('transform', `translate(${margin.left}, ${margin.top})`);
    
    // RÉCUPÉRER LES DONNÉES DE DURÉE DES PROGRAMMES
    // Récupérer les données de durée totale en heures pour chaque chaîne
    function getProgramDurationData(year) {
        // Récupérer les données de durée pour l'année spécifiée
        // Si aucune donnée n'est disponible pour cette année, utiliser les données de 2019 par défaut
        const durationData = PROGRAM_DURATION_DATA[year] || PROGRAM_DURATION_DATA['2019'] || {};
        
        return durationData;
    }
    
    // Récupérer les données de durée
    const durationData = getProgramDurationData(year);
    
    // Préparer les données avec les vraies durées
    const bubbleData = data.datasets.map(dataset => {
        const channelName = dataset.label;
        const womenRate = dataset.data[0]?.y || 0;
        
        // Obtenir la durée réelle pour cette chaîne
        // Chercher d'abord le nom exact, puis chercher des correspondances partielles
        let duration = durationData[channelName];
        
        if (duration === undefined) {
            // Essayer de trouver une correspondance partielle
            for (const [key, value] of Object.entries(durationData)) {
                if (channelName.includes(key) || key.includes(channelName)) {
                    duration = value;
                    break;
                }
            }
        }
        
        // Valeur par défaut si non trouvé (basée sur la moyenne)
        if (duration === undefined) {
            const allDurations = Object.values(durationData);
            const avgDuration = allDurations.length > 0 ? 
                allDurations.reduce((a, b) => a + b, 0) / allDurations.length : 1500;
            duration = avgDuration * (0.8 + Math.random() * 0.4); // Variation aléatoire autour de la moyenne
        }
        
        return {
            label: channelName,
            x: duration, // ← UTILISATION DE LA DURÉE RÉELLE EN HEURES
            y: womenRate,
            r: dataset.data[0]?.r || 10,
            channel: channelName,
            color: dataset.backgroundColor || getColorByChannel(channelName),
            duration: duration // Stocker aussi la durée pour référence
        };
    });
    
    // Fonction pour obtenir une couleur basée sur la chaîne
    function getColorByChannel(channel) {
        // Palette TV rétro cohérente
        const channelColors = {
            // Chaînes principales
            'TF1': '#ff6b8b',
            'France 2': '#4a9eff',
            'France 3': '#50c878',
            'M6': '#ff9f43',
            'ARTE': '#a370f7',
            'Canal+': '#00d4ff',
            'France 24': '#ff4757',
            'BFM TV': '#2ed573',
            // Chaînes par défaut
            'default': '#5352ed'
        };
        
        for (const [key, color] of Object.entries(channelColors)) {
            if (channel.includes(key)) {
                return color;
            }
        }
        
        // Générer une couleur basée sur le hash de la chaîne pour la cohérence
        let hash = 0;
        for (let i = 0; i < channel.length; i++) {
            hash = channel.charCodeAt(i) + ((hash << 5) - hash);
        }
        
        const hue = hash % 360;
        return `hsl(${hue}, 70%, 60%)`;
    }
    
    // Calculer les domaines avec les vraies données de durée
    const xValues = bubbleData.map(d => d.x).filter(v => !isNaN(v) && v > 0);
    const yValues = bubbleData.map(d => d.y).filter(v => !isNaN(v) && v > 0);
    
    if (xValues.length === 0 || yValues.length === 0) {
        container.innerHTML = '<div class="no-data">Données insuffisantes pour afficher le graphique</div>';
        return;
    }
    
    // Calculer les domaines avec marges
    const xMin = d3.min(xValues);
    const xMax = d3.max(xValues);
    const yMin = Math.max(0, d3.min(yValues) - 5);
    const yMax = Math.min(100, d3.max(yValues) + 5);
    
    // Ajouter 10% de marge pour éviter que les bulles touchent les bords
    const xRange = xMax - xMin;
    const yRange = yMax - yMin;
    
    const xDomain = [
        Math.max(0, xMin - (xRange * 0.1)),
        xMax + (xRange * 0.1)
    ];
    
    const yDomain = [
        Math.max(0, yMin - (yRange * 0.1)),
        Math.min(100, yMax + (yRange * 0.1))
    ];
    
    // Échelles
    const xScale = d3.scaleLinear()
        .domain(xDomain)
        .range([0, width])
        .nice();
    
    const yScale = d3.scaleLinear()
        .domain(yDomain)
        .range([height, 0])
        .nice();
    
    // Échelle pour les rayons (basée sur le volume relatif)
    const rValues = bubbleData.map(d => d.r).filter(v => !isNaN(v) && v > 0);
    const rMin = d3.min(rValues);
    const rMax = d3.max(rValues);
    
    const rScale = d3.scaleSqrt()
        .domain([rMin, rMax])
        .range([15, 50]); // Taille minimale et maximale des bulles
    
    // Axe X avec formatage des heures - MODIFIÉ POUR AFFICHER "h"
    const xAxis = d3.axisBottom(xScale)
        .ticks(8)
        .tickSizeOuter(0)
        .tickFormat(d => {
            // Formater en heures avec le suffixe "h" comme demandé
            return `${d.toLocaleString('fr-FR')} h`;
        });
    
    svg.append('g')
        .attr('class', 'x-axis')
        .attr('transform', `translate(0, ${height})`)
        .call(xAxis)
        .selectAll('text')
        .style('fill', TV_COLORS.text)
        .style('font-family', 'Segoe UI')
        .style('font-size', '11px')
        .style('font-weight', '400');
    
    // Titre axe X - Style inspiré du premier graphique
    svg.append('text')
        .attr('class', 'axis-title')
        .attr('x', width / 2)
        .attr('y', height + margin.bottom - 25)
        .attr('text-anchor', 'middle')
        .style('fill', TV_COLORS.text)
        .style('font-family', 'Segoe UI')
        .style('font-size', '14px')
        .style('font-weight', '600')
        .style('letter-spacing', '0.5px')
        .text('Durée totale des programmes (heures)'); // ← TITRE MIS À JOUR
    
    // Axe Y avec style amélioré
    const yAxis = d3.axisLeft(yScale)
        .ticks(10)
        .tickFormat(d => `${d}%`)
        .tickSizeOuter(0);
    
    svg.append('g')
        .attr('class', 'y-axis')
        .call(yAxis)
        .selectAll('text')
        .style('fill', TV_COLORS.text)
        .style('font-family', 'Segoe UI')
        .style('font-size', '11px')
        .style('font-weight', '400');
    
    // Titre axe Y - Style inspiré du premier graphique
    svg.append('text')
        .attr('class', 'axis-title')
        .attr('transform', 'rotate(-90)')
        .attr('x', -height / 2)
        .attr('y', -margin.left + 30)
        .attr('text-anchor', 'middle')
        .style('fill', TV_COLORS.text)
        .style('font-family', 'Segoe UI')
        .style('font-size', '14px')
        .style('font-weight', '600')
        .style('letter-spacing', '0.5px')
        .text('Taux de parole des femmes (%)');
    
    // Grille améliorée
    svg.append('g')
        .attr('class', 'grid y-grid')
        .call(d3.axisLeft(yScale)
            .tickSize(-width)
            .tickFormat('')
            .ticks(10))
        .selectAll('line')
        .style('stroke', 'rgba(255, 255, 255, 0.15)')
        .style('stroke-width', 1)
        .style('stroke-dasharray', '4,4');
    
    svg.append('g')
        .attr('class', 'grid x-grid')
        .attr('transform', `translate(0, ${height})`)
        .call(d3.axisBottom(xScale)
            .tickSize(-height)
            .tickFormat('')
            .ticks(8))
        .selectAll('line')
        .style('stroke', 'rgba(255, 255, 255, 0.15)')
        .style('stroke-width', 1)
        .style('stroke-dasharray', '4,4');
    
    // Ligne de référence pour la parité (50%)
    if (yDomain[0] <= 50 && yDomain[1] >= 50) {
        svg.append('line')
            .attr('class', 'parity-line')
            .attr('x1', 0)
            .attr('y1', yScale(50))
            .attr('x2', width)
            .attr('y2', yScale(50))
            .style('stroke', 'rgba(255, 255, 255, 0.3)')
            .style('stroke-width', 2)
            .style('stroke-dasharray', '10,5');
        
        svg.append('text')
            .attr('class', 'parity-label')
            .attr('x', width - 10)
            .attr('y', yScale(50) - 8)
            .attr('text-anchor', 'end')
            .style('fill', 'rgba(255, 255, 255, 0.6)')
            .style('font-family', 'Segoe UI')
            .style('font-size', '11px')
            .style('font-weight', '500')
            .text('Parité (50%)');
    }
    
    // Ajouter des filtres SVG pour les effets
    const defs = svg.append('defs');
    
    // Filtre de brillance pour les bulles
    const glowFilter = defs.append('filter')
        .attr('id', 'bubble-glow')
        .attr('x', '-50%')
        .attr('y', '-50%')
        .attr('width', '200%')
        .attr('height', '200%');
    
    glowFilter.append('feGaussianBlur')
        .attr('stdDeviation', '2')
        .attr('result', 'coloredBlur');
    
    const feMerge = glowFilter.append('feMerge');
    feMerge.append('feMergeNode').attr('in', 'coloredBlur');
    feMerge.append('feMergeNode').attr('in', 'SourceGraphic');
    
    // Dessiner les bulles avec effets
    const bubbles = svg.selectAll('.bubble')
        .data(bubbleData)
        .enter()
        .append('g')
        .attr('class', 'bubble-group')
        .attr('transform', d => `translate(${xScale(d.x)}, ${yScale(d.y)})`);
    
    // Fonction pour calculer la taille de police du label
    function getLabelFontSize(r) {
        const radius = rScale(r);
        return Math.max(9, Math.min(12, radius / 4)) + 'px';
    }
    
    // Cercles avec effet de brillance
    bubbles.append('circle')
        .attr('class', 'bubble')
        .attr('r', d => rScale(d.r))
        .style('fill', d => d.color)
        .style('fill-opacity', 0.8)
        .style('stroke', d => d3.color(d.color).brighter(0.5))
        .style('stroke-width', 2)
        .style('cursor', 'pointer')
        .style('filter', 'url(#bubble-glow)')
        .on('mouseover', function(event, d) {
            // Agrandir la bulle
            d3.select(this)
                .transition()
                .duration(200)
                .attr('r', rScale(d.r) * 1.3)
                .style('fill-opacity', 1)
                .style('stroke-width', 3);
            
            // Mettre en avant le label
            d3.select(this.parentNode).select('.bubble-label')
                .transition()
                .duration(200)
                .style('font-weight', '700')
                .style('font-size', '13px');
            
            // Afficher le tooltip MIS À JOUR
            showBubbleTooltip(event, d);
        })
        .on('mouseout', function(event, d) {
            // Réduire la bulle
            d3.select(this)
                .transition()
                .duration(200)
                .attr('r', rScale(d.r))
                .style('fill-opacity', 0.8)
                .style('stroke-width', 2);
            
            // Réduire le label
            d3.select(this.parentNode).select('.bubble-label')
                .transition()
                .duration(200)
                .style('font-weight', '600')
                .style('font-size', d => getLabelFontSize(d.r));
            
            // Cacher le tooltip
            hideBubbleTooltip();
        });
    
    // Labels positionnés intelligemment
    bubbles.append('text')
        .attr('class', 'bubble-label')
        .attr('text-anchor', 'middle')
        .attr('dy', '.3em')
        .text(d => {
            // Abréger les noms longs
            const name = d.label;
            if (name.length > 12) {
                return name.substring(0, 10) + '...';
            }
            return name;
        })
        .style('fill', '#ffffff')
        .style('font-family', 'Segoe UI')
        .style('font-size', d => getLabelFontSize(d.r))
        .style('font-weight', '600')
        .style('pointer-events', 'none')
        .style('text-shadow', '1px 1px 2px rgba(0,0,0,0.5)');
    
    // Titre principal du graphique
    svg.append('text')
        .attr('class', 'chart-title')
        .attr('x', width / 2)
        .attr('y', -30)
        .attr('text-anchor', 'middle')
        .style('fill', TV_COLORS.text)
        .style('font-family', 'Segoe UI')
        .style('font-size', '18px')
        .style('font-weight', '700')
        .style('letter-spacing', '0.5px')
        .text(`Volume vs taux d'expression - ${year}`);
    
    // Sous-titre descriptif
    svg.append('text')
        .attr('class', 'chart-subtitle')
        .attr('x', width / 2)
        .attr('y', -8)
        .attr('text-anchor', 'middle')
        .style('fill', 'rgba(255, 255, 255, 0.8)')
        .style('font-family', 'Segoe UI')
        .style('font-size', '12px')
        .style('font-weight', '400')
        .style('font-style', 'italic')
        .text("Volume d'heures de programme vs taux de parole des femmes");
    
    // Légende interactive avec style TV rétro
    const legend = svg.append('g')
        .attr('class', 'legend')
        .attr('transform', `translate(${width + 20}, 0)`);
    
    // Titre de la légende
    legend.append('text')
        .attr('class', 'legend-title')
        .attr('x', 0)
        .attr('y', -10)
        .style('fill', TV_COLORS.text)
        .style('font-family', 'Segoe UI')
        .style('font-size', '13px')
        .style('font-weight', '600')
        .text('Chaînes');
    
    const legendItems = legend.selectAll('.legend-item')
        .data(bubbleData)
        .enter()
        .append('g')
        .attr('class', 'legend-item')
        .attr('transform', (d, i) => `translate(0, ${i * 28})`)
        .style('cursor', 'pointer')
        .on('mouseover', function(event, d) {
            // Mettre en surbrillance la bulle correspondante
            svg.selectAll('.bubble-group')
                .filter(b => b.label === d.label)
                .select('.bubble')
                .transition()
                .duration(200)
                .attr('r', rScale(d.r) * 1.3)
                .style('fill-opacity', 1);
            
            svg.selectAll('.bubble-group')
                .filter(b => b.label !== d.label)
                .select('.bubble')
                .transition()
                .duration(200)
                .style('fill-opacity', 0.3);
        })
        .on('mouseout', function() {
            // Restaurer l'opacité normale
            svg.selectAll('.bubble')
                .transition()
                .duration(200)
                .attr('r', d => rScale(d.r))
                .style('fill-opacity', 0.8);
        })
        .on('click', function(event, d) {
            // Toggle de la visibilité
            const bubbleGroup = svg.selectAll('.bubble-group')
                .filter(b => b.label === d.label);
            
            const isVisible = bubbleGroup.style('display') !== 'none';
            
            bubbleGroup.transition()
                .duration(300)
                .style('display', isVisible ? 'none' : 'block');
            
            // Mettre à jour la légende
            d3.select(this).select('rect')
                .transition()
                .duration(300)
                .style('opacity', isVisible ? 0.3 : 1);
            
            d3.select(this).select('text')
                .transition()
                .duration(300)
                .style('opacity', isVisible ? 0.3 : 1)
                .style('text-decoration', isVisible ? 'line-through' : 'none');
            
            // Effet glitch
            triggerGlitchEffect(150);
        });
    
    // Cases de couleur avec style TV
    legendItems.append('rect')
        .attr('width', 16)
        .attr('height', 16)
        .attr('rx', 3)
        .style('fill', d => d.color)
        .style('stroke', '#ffffff')
        .style('stroke-width', 1.5)
        .style('stroke-opacity', 0.8);
    
    // Noms des chaînes
    legendItems.append('text')
        .attr('x', 22)
        .attr('y', 12)
        .text(d => d.label)
        .style('fill', TV_COLORS.text)
        .style('font-family', 'Segoe UI')
        .style('font-size', '11px')
        .style('user-select', 'none')
        .style('font-weight', '500');
    
    // Indicateur de taille des bulles
    const sizeLegend = svg.append('g')
        .attr('class', 'size-legend')
        .attr('transform', `translate(${width + 20}, ${bubbleData.length * 28 + 40})`);
    
    sizeLegend.append('text')
        .attr('x', 0)
        .attr('y', -5)
        .style('fill', TV_COLORS.text)
        .style('font-family', 'Segoe UI')
        .style('font-size', '12px')
        .style('font-weight', '600')
        .text('Taille = Volume relatif');
    
    // Exemples de tailles
    const sizeExamples = [
        { label: 'Petit', r: rMin },
        { label: 'Moyen', r: (rMin + rMax) / 2 },
        { label: 'Grand', r: rMax }
    ];
    
    const sizeItems = sizeLegend.selectAll('.size-item')
        .data(sizeExamples)
        .enter()
        .append('g')
        .attr('class', 'size-item')
        .attr('transform', (d, i) => `translate(0, ${i * 30})`);
    
    sizeItems.append('circle')
        .attr('cx', 8)
        .attr('cy', 8)
        .attr('r', d => rScale(d.r))
        .style('fill', 'rgba(255, 255, 255, 0.2)')
        .style('stroke', TV_COLORS.text)
        .style('stroke-width', 1);
    
    sizeItems.append('text')
        .attr('x', 25)
        .attr('y', 12)
        .text(d => d.label)
        .style('fill', TV_COLORS.text)
        .style('font-family', 'Segoe UI')
        .style('font-size', '10px')
        .style('font-weight', '400');
    
    // Fonctions pour le tooltip des bulles MIS À JOUR
    function showBubbleTooltip(event, d) {
        const tooltip = d3.select('#d3-bubble-tooltip');
        
        if (tooltip.empty()) {
            d3.select('body')
                .append('div')
                .attr('id', 'd3-bubble-tooltip')
                .style('position', 'absolute')
                .style('background', 'rgba(0, 0, 0, 0.95)')
                .style('color', '#ffffff')
                .style('padding', '15px')
                .style('border-radius', '8px')
                .style('border', `2px solid ${d.color}`)
                .style('font-family', 'Segoe UI')
                .style('font-size', '12px')
                .style('pointer-events', 'none')
                .style('z-index', '1000')
                .style('min-width', '220px')
                .style('max-width', '280px')
                .style('box-shadow', '0 4px 20px rgba(0,0,0,0.5)')
                .style('backdrop-filter', 'blur(5px)');
        }
        
        const tooltipDiv = d3.select('#d3-bubble-tooltip');
        
        // Calculer les statistiques pour cette bulle
        const menRate = 100 - d.y;
        const parityScore = Math.abs(50 - d.y);
        const status = d.y >= 45 ? 'Bon' : d.y >= 35 ? 'Moyen' : 'Faible';
        const statusColor = d.y >= 45 ? '#2ed573' : d.y >= 35 ? '#ff9f43' : '#ff4757';
        
        tooltipDiv.html(`
            <div style="display: flex; align-items: center; margin-bottom: 10px;">
                <div style="width: 12px; height: 12px; border-radius: 50%; background: ${d.color}; margin-right: 8px;"></div>
                <div style="font-weight: 700; font-size: 14px; color: ${d.color}">${d.label}</div>
            </div>
            <div style="margin-bottom: 8px; padding-bottom: 8px; border-bottom: 1px solid rgba(255,255,255,0.1);">
                <div style="display: flex; justify-content: space-between; margin-bottom: 4px;">
                    <span style="opacity: 0.8;">Durée totale:</span>
                    <span style="font-weight: 600;">${d.x.toLocaleString('fr-FR')} h</span> <!-- Formaté avec "h" -->
                </div>
                <div style="display: flex; justify-content: space-between;">
                    <span style="opacity: 0.8;">Taux de parole:</span>
                    <span style="font-weight: 600;">${d.y.toFixed(1)}% femmes</span>
                </div>
            </div>
            <div style="margin-bottom: 8px;">
                <div style="display: flex; justify-content: space-between; margin-bottom: 4px;">
                    <span style="opacity: 0.8;">Hommes:</span>
                    <span>${menRate.toFixed(1)}%</span>
                </div>
                <div style="display: flex; justify-content: space-between; margin-bottom: 4px;">
                    <span style="opacity: 0.8;">Écart à la parité:</span>
                    <span>${parityScore.toFixed(1)} points</span>
                </div>
            </div>
            <div style="padding: 6px 10px; background: rgba(255,255,255,0.1); border-radius: 4px; text-align: center;">
                <span style="color: ${statusColor}; font-weight: 600; font-size: 11px;">${status} taux de parité</span>
            </div>
        `);
        
        const tooltipWidth = tooltipDiv.node().offsetWidth;
        const tooltipHeight = tooltipDiv.node().offsetHeight;
        
        // Positionner le tooltip intelligemment pour éviter les bords
        let left = event.pageX + 15;
        let top = event.pageY - 15;
        
        if (left + tooltipWidth > window.innerWidth) {
            left = event.pageX - tooltipWidth - 15;
        }
        
        if (top + tooltipHeight > window.innerHeight) {
            top = event.pageY - tooltipHeight - 15;
        }
        
        tooltipDiv.style('left', left + 'px')
                 .style('top', top + 'px')
                 .style('opacity', 1);
    }
    
    function hideBubbleTooltip() {
        d3.select('#d3-bubble-tooltip')
            .transition()
            .duration(200)
            .style('opacity', 0);
    }
    
    // Retourner le nœud SVG
    return svg.node();
}

// ===== GRAPHIQUE RADIAL STACKED BAR CHART (3ÈME GRAPHIQUE) - VERSION OPTIMISÉE ET CENTRÉE =====
function renderRadialStackedChartD3(containerId, config) {
    const container = document.getElementById(containerId);
    if (!container) return null;
    
    // Effacer le contenu précédent
    container.innerHTML = '';
    
    // DIMENSIONS OPTIMISÉES POUR LA LISIBILITÉ
    const containerWidth = container.clientWidth;
    const containerHeight = 500; // Hauteur fixe pour contrôler l'affichage
    
    // Largeur maximale raisonnable pour éviter l'étirement
    const width = Math.min(containerWidth, 700); 
    const height = Math.min(containerHeight, 500);
    
    // Rayons ajustés pour la lisibilité
    const innerRadius = 60;  // Réduit pour plus d'espace au centre
    const outerRadius = Math.min(width, height) / 2 - 50; // Plus de marge
    
    // Récupérer l'année sélectionnée
    const year = document.getElementById('currentYear')?.textContent || '2019';
    
    // Récupérer les données réelles
    const filteredData = getFilteredData(year, currentMediaFilter);
    
    // Si pas de données, afficher un message
    if (!filteredData.labels || filteredData.labels.length === 0) {
        container.innerHTML = '<div class="no-data" style="text-align: center; color: #fff; padding: 20px; font-family: Segoe UI;">Aucune donnée disponible pour cette sélection</div>';
        return;
    }
    
    // Préparer les données pour le graphique radial
    const hours = Array.from({length: 24}, (_, i) => i); // 0 à 23h
    
    // Limiter à 7 chaînes maximum pour la lisibilité (comme dans l'image)
    const maxChannels = 7;
    const channels = filteredData.labels.slice(0, maxChannels);
    const womenRates = filteredData.femaleData.slice(0, maxChannels);
    
    // Palette de couleurs RVB améliorée - correspondant aux couleurs de l'image
    const RGB_COLORS = [
        "#FF6B8B", // Rose (Chambre d'activité)
        "#FF8C00", // Orange (Animaux)
        "#FFDD00", // Jaune (BFM TV)
        "#00CC44", // Vert (Canal+)
        "#00DDFF", // Cyan clair (Canal+ Sport)
        "#4488FF", // Bleu (Chasse et pêche)
        "#AA44FF"  // Violet (Chérie 25)
    ];
    
    // Créer les données structurées
    const data = [];
    
    // Créer les données pour chaque heure avec variation réaliste
    hours.forEach(hour => {
        channels.forEach((channel, channelIndex) => {
            // Utiliser les données réelles avec variation horaire
            const baseRate = womenRates[channelIndex] || 30;
            // Variation basée sur l'heure (plus bas la nuit, plus haut en journée)
            const hourFactor = Math.sin((hour - 6) / 12 * Math.PI) * 15; // Pic vers 15h
            const womenRate = Math.max(10, Math.min(90, baseRate + hourFactor));
            
            data.push({
                hour: hour,
                hourLabel: `${hour}h`,
                channel: channel,
                womenRate: womenRate,
                menRate: 100 - womenRate
            });
        });
    });
    
    // Créer le SVG principal avec des dimensions fixes et centré
    const svg = d3.select(`#${containerId}`)
        .append('svg')
        .attr('width', '100%')
        .attr('height', '100%')
        .attr('viewBox', [-width / 2, -height / 2, width, height])
        .attr('style', `
            width: 100%; 
            height: ${height}px; 
            max-width: 700px; 
            margin: 0 auto; 
            display: block;
            overflow: visible;
        `)
        .attr('preserveAspectRatio', 'xMidYMid meet');
    
    // ===== TITRE ET DESCRIPTION =====
    const titleGroup = svg.append("g")
        .attr("class", "title-group")
        .attr("transform", `translate(0, ${-height/2 + 40})`);
    
    // Titre principal
    titleGroup.append("text")
        .attr("class", "chart-title")
        .attr("x", 0)
        .attr("y", 0)
        .attr("text-anchor", "middle")
        .style("fill", "#ffffff")
        .style("font-family", "Segoe UI, Arial, sans-serif")
        .style("font-size", "18px")
        .style("font-weight", "700")
        .style("letter-spacing", "0.3px")
        .style("text-shadow", "0 2px 4px rgba(0,0,0,0.5)")
        .text("Taux de parole des femmes par créneau horaire");
    
    // Sous-titre descriptif
    const avgRate = calculateMean(filteredData.femaleData).toFixed(1);
    const descriptionText = `En ${year}, les femmes s'expriment en moyenne à hauteur de ${avgRate}% sur ${channels.length} chaînes`;
    
    titleGroup.append("text")
        .attr("class", "chart-description")
        .attr("x", 0)
        .attr("y", 25)
        .attr("text-anchor", "middle")
        .style("fill", "rgba(255, 255, 255, 0.85)")
        .style("font-family", "Segoe UI, Arial, sans-serif")
        .style("font-size", "13px")
        .style("font-weight", "400")
        .style("font-style", "italic")
        .text(descriptionText);
    
    // ===== PRÉPARATION DES DONNÉES POUR LE STACKING =====
    // Regrouper par heure
    const hoursData = d3.group(data, d => d.hour);
    
    // Créer les séries pour chaque chaîne
    const channelNames = [...new Set(data.map(d => d.channel))];
    
    // Stack generator
    const stack = d3.stack()
        .keys(channelNames)
        .value(([hour, values], channel) => {
            const item = values.find(v => v.channel === channel);
            return item ? item.womenRate : 0;
        });
    
    const series = stack(Array.from(hoursData));
    
    // ===== ÉCHELLES =====
    // Échelle angulaire
    const xScale = d3.scaleBand()
        .domain(hours.map(h => h))
        .range([0, 2 * Math.PI])
        .align(0)
        .padding(0.05);
    
    // Échelle radiale
    const maxValue = d3.max(series, s => d3.max(s, d => d[1]));
    const yScale = d3.scaleRadial()
        .domain([0, Math.max(maxValue, 80)])
        .range([innerRadius, outerRadius]);
    
    // Palette de couleurs
    const colorScale = d3.scaleOrdinal()
        .domain(channelNames)
        .range(RGB_COLORS.slice(0, channelNames.length))
        .unknown("#888888");
    
    // ===== DESSIN DES BARRES RADIALES =====
    // Fonction d'arc
    const arc = d3.arc()
        .innerRadius(d => yScale(d[0]))
        .outerRadius(d => yScale(d[1]))
        .startAngle(d => xScale(d.data[0]) || 0)
        .endAngle(d => (xScale(d.data[0]) || 0) + xScale.bandwidth())
        .padAngle(0.01)
        .padRadius(innerRadius);
    
    // Groupe pour les barres
    const barsGroup = svg.append("g")
        .attr("class", "bars-group");
    
    // Dessiner chaque série
    const seriesGroups = barsGroup.selectAll(".series")
        .data(series)
        .enter()
        .append("g")
        .attr("class", "series")
        .attr("fill", d => colorScale(d.key))
        .style("opacity", 0.85);
    
    // Ajouter les barres
    seriesGroups.selectAll(".bar")
        .data(d => d.map(point => ({...point, key: d.key})))
        .enter()
        .append("path")
        .attr("class", "bar")
        .attr("d", arc)
        .style("stroke", "#ffffff")
        .style("stroke-width", 1)
        .style("stroke-opacity", 0.4)
        .style("opacity", 0.9)
        .on("mouseover", function(event, d) {
            // Surbrillance
            d3.select(this)
                .style("stroke-width", 2)
                .style("stroke-opacity", 0.8)
                .style("filter", "brightness(1.3)")
                .style("opacity", 1);
            
            // Afficher le tooltip
            showRadialTooltip(event, d);
        })
        .on("mouseout", function(event, d) {
            // Retirer la surbrillance
            d3.select(this)
                .style("stroke-width", 1)
                .style("stroke-opacity", 0.4)
                .style("filter", "brightness(1)")
                .style("opacity", 0.9);
            
            // Cacher le tooltip
            hideRadialTooltip();
        })
        .append("title")
        .text(d => {
            const hour = d.data[0];
            const channel = d.key;
            const value = d[1] - d[0];
            return `${hour}h - ${channel}: ${value.toFixed(1)}% femmes`;
        });
    
    // ===== AXE DES HEURES =====
    const hourAxisGroup = svg.append("g")
        .attr("class", "hour-axis")
        .attr("text-anchor", "middle");
    
    // Heures principales à afficher - SUPPRIMÉ LES HEURES DU CENTRE
    // On garde uniquement les heures autour du cercle
    const displayHours = [0, 6, 12, 18];
    
    displayHours.forEach(hour => {
        const angle = xScale(hour) + xScale.bandwidth() / 2;
        const group = hourAxisGroup.append("g")
            .attr("transform", `rotate(${(angle * 180 / Math.PI - 90)}) translate(${outerRadius + 20},0)`);
        
        // Ligne indicateur (plus longue pour pointer vers le cercle)
        group.append("line")
            .attr("x1", -15)
            .attr("x2", 0)
            .attr("stroke", "#ffffff")
            .attr("stroke-opacity", 0.6)
            .attr("stroke-width", 1.5);
        
        // Texte de l'heure (positionné à l'extérieur)
        group.append("text")
            .attr("transform", `rotate(${angle > Math.PI ? -90 : 90})`)
            .attr("y", angle > Math.PI ? -20 : 15)
            .style("fill", "#ffffff")
            .style("font-family", "Segoe UI")
            .style("font-size", "12px")
            .style("font-weight", "600")
            .style("text-shadow", "0 1px 2px rgba(0,0,0,0.8)")
            .text(`${hour}h`);
    });
    
    // ===== AXE RADIAL =====
    const radialAxisGroup = svg.append("g")
        .attr("class", "radial-axis")
        .attr("text-anchor", "middle");
    
    // Cercles concentriques
    const radialTicks = yScale.ticks(4);
    
    radialAxisGroup.selectAll(".grid-circle")
        .data(radialTicks.slice(1))
        .enter()
        .append("circle")
        .attr("r", yScale)
        .attr("fill", "none")
        .attr("stroke", "rgba(255, 255, 255, 0.15)")
        .attr("stroke-width", 1)
        .attr("stroke-dasharray", "3,2");
    
    // Labels des cercles concentriques
    radialAxisGroup.selectAll(".grid-label")
        .data(radialTicks)
        .enter()
        .append("text")
        .attr("y", d => -yScale(d))
        .attr("dy", "0.35em")
        .style("fill", "#ffffff")
        .style("font-family", "Segoe UI")
        .style("font-size", "11px")
        .style("font-weight", "500")
        .style("text-shadow", "0 1px 2px rgba(0,0,0,0.8)")
        .text(d => d + '%');
    
    // ===== LÉGENDE DES CHAÎNES - POSITIONNÉE À DROITE =====
    const legendWidth = 180;
    const legendX = width/2 - 20;
    const legendY = -height/2 + 80;
    
    const legend = svg.append("g")
        .attr("class", "radial-legend")
        .attr("transform", `translate(${legendX}, ${legendY})`);
    
    // Titre de la légende
    legend.append("text")
        .attr("class", "legend-title")
        .attr("x", 0)
        .attr("y", -15)
        .style("fill", "#ffffff")
        .style("font-family", "Segoe UI")
        .style("font-size", "13px")
        .style("font-weight", "600")
        .style("text-shadow", "0 1px 2px rgba(0,0,0,0.8)")
        .text("Chaînes");
    
    // Items de légende
    const legendItems = legend.selectAll(".legend-item")
        .data(channelNames)
        .enter()
        .append("g")
        .attr("class", "legend-item")
        .attr("transform", (d, i) => `translate(0, ${i * 22})`)
        .style("cursor", "pointer")
        .on("mouseover", function(event, channel) {
            // Surbrillance de la chaîne
            barsGroup.selectAll(".bar")
                .filter(d => d.key === channel)
                .style("stroke-width", 2)
                .style("stroke-opacity", 0.8)
                .style("filter", "brightness(1.3)");
        })
        .on("mouseout", function(event, channel) {
            // Retirer la surbrillance
            barsGroup.selectAll(".bar")
                .style("stroke-width", 1)
                .style("stroke-opacity", 0.4)
                .style("filter", "brightness(1)");
        })
        .on("click", function(event, channel) {
            // Toggle de la visibilité
            const bars = barsGroup.selectAll(".bar")
                .filter(d => d.key === channel);
            
            const isVisible = bars.style("opacity") !== "0";
            const newOpacity = isVisible ? 0 : 0.9;
            
            bars.transition()
                .duration(300)
                .style("opacity", newOpacity);
            
            // Mettre à jour la légende
            d3.select(this).select("rect")
                .transition()
                .duration(300)
                .style("opacity", isVisible ? 0.3 : 1);
            
            d3.select(this).select("text")
                .transition()
                .duration(300)
                .style("opacity", isVisible ? 0.3 : 1);
        });
    
    // Carrés de couleur
    legendItems.append("rect")
        .attr("width", 14)
        .attr("height", 14)
        .attr("fill", colorScale)
        .attr("rx", 3)
        .attr("ry", 3)
        .style("stroke", "#ffffff")
        .style("stroke-width", 1)
        .style("stroke-opacity", 0.3);
    
    // Noms des chaînes (tronqués si trop longs)
    legendItems.append("text")
        .attr("x", 20)
        .attr("y", 11)
        .attr("dy", "0.35em")
        .style("fill", "#ffffff")
        .style("font-family", "Segoe UI")
        .style("font-size", "11px")
        .style("font-weight", "500")
        .style("text-shadow", "0 1px 2px rgba(0,0,0,0.8)")
        .text(d => {
            // Tronquer intelligemment comme dans l'image
            if (d.length > 20) {
                return d.substring(0, 18) + '...';
            } else if (d.length > 15) {
                return d.substring(0, 14) + '..';
            }
            return d;
        });
    
    // ===== ZONE CENTRALE DU GRAPHIQUE =====
    // Créer un groupe pour la zone centrale (sans les heures du centre)
    const centerGroup = svg.append("g")
        .attr("class", "center-group");
    
    // Ajouter un cercle central décoratif
    centerGroup.append("circle")
        .attr("cx", 0)
        .attr("cy", 0)
        .attr("r", innerRadius - 10)
        .attr("fill", "rgba(0, 0, 0, 0.2)")
        .attr("stroke", "rgba(255, 255, 255, 0.1)")
        .attr("stroke-width", 1);
    
    // Ajouter le texte de pourcentage moyen au centre
    centerGroup.append("text")
        .attr("class", "center-percentage")
        .attr("x", 0)
        .attr("y", -5)
        .attr("text-anchor", "middle")
        .style("fill", "#ffffff")
        .style("font-family", "Segoe UI")
        .style("font-size", "24px")
        .style("font-weight", "700")
        .style("text-shadow", "0 2px 4px rgba(0,0,0,0.8)")
        .text(`${avgRate}%`);
    
    centerGroup.append("text")
        .attr("class", "center-label")
        .attr("x", 0)
        .attr("y", 20)
        .attr("text-anchor", "middle")
        .style("fill", "rgba(255, 255, 255, 0.8)")
        .style("font-family", "Segoe UI")
        .style("font-size", "12px")
        .style("font-weight", "500")
        .text("moyenne femmes");
    
    // ===== NOTE D'INFORMATION =====
    svg.append("text")
        .attr("class", "chart-note")
        .attr("x", 0)
        .attr("y", height/2 - 25)
        .attr("text-anchor", "middle")
        .style("fill", "rgba(255, 255, 255, 0.6)")
        .style("font-family", "Segoe UI")
        .style("font-size", "11px")
        .style("font-style", "italic")
        .style("font-weight", "400")
        .text("Chaque créneau horaire (0h-23h) montre le taux de parole par chaîne");
    
    // ===== FONCTIONS POUR LES TOOLTIPS =====
    function showRadialTooltip(event, d) {
        // Créer le tooltip
        let tooltip = d3.select('#radial-tooltip');
        
        if (tooltip.empty()) {
            tooltip = d3.select('body')
                .append('div')
                .attr('id', 'radial-tooltip')
                .style('position', 'absolute')
                .style('background', 'rgba(26, 26, 26, 0.98)')
                .style('color', '#ffffff')
                .style('padding', '12px')
                .style('border-radius', '8px')
                .style('border', '2px solid rgba(0, 168, 255, 0.6)')
                .style('font-family', 'Segoe UI, Arial, sans-serif')
                .style('font-size', '12px')
                .style('pointer-events', 'none')
                .style('z-index', '10000')
                .style('min-width', '220px')
                .style('max-width', '280px')
                .style('box-shadow', '0 4px 20px rgba(0,0,0,0.5)')
                .style('backdrop-filter', 'blur(5px)')
                .style('opacity', 0);
        }
        
        const hour = d.data[0];
        const channel = d.key;
        const womenValue = d[1] - d[0];
        const menValue = 100 - womenValue;
        const color = colorScale(channel);
        
        const tooltipHTML = `
            <div style="display: flex; align-items: center; margin-bottom: 10px;">
                <div style="width: 12px; height: 12px; border-radius: 50%; background: ${color}; margin-right: 8px;"></div>
                <div style="font-weight: 700; font-size: 14px; color: ${color}">${channel}</div>
            </div>
            <div style="margin-bottom: 8px; padding-bottom: 8px; border-bottom: 1px solid rgba(255,255,255,0.1);">
                <div style="display: flex; justify-content: space-between; margin-bottom: 5px;">
                    <span style="opacity: 0.8;">Créneau horaire:</span>
                    <span style="font-weight: 600;">${hour}h - ${hour+1}h</span>
                </div>
                <div style="display: flex; justify-content: space-between;">
                    <span style="opacity: 0.8;">Taux femmes:</span>
                    <span style="font-weight: 600; color: ${color}">${womenValue.toFixed(1)}%</span>
                </div>
            </div>
            <div style="margin-bottom: 8px;">
                <div style="display: flex; justify-content: space-between; margin-bottom: 4px;">
                    <span style="opacity: 0.8;">Taux hommes:</span>
                    <span>${menValue.toFixed(1)}%</span>
                </div>
            </div>
            <div style="font-size: 10px; opacity: 0.7; margin-top: 8px;">
                Cliquez sur la légende pour masquer/afficher une chaîne
            </div>
        `;
        
        tooltip.html(tooltipHTML)
            .style('opacity', 0)
            .transition()
            .duration(200)
            .style('opacity', 1);
        
        // Positionnement intelligent
        const tooltipWidth = tooltip.node().offsetWidth;
        const tooltipHeight = tooltip.node().offsetHeight;
        
        let left = event.pageX + 15;
        let top = event.pageY - 15;
        
        // Éviter les bords de l'écran
        if (left + tooltipWidth > window.innerWidth) {
            left = event.pageX - tooltipWidth - 15;
        }
        
        if (top + tooltipHeight > window.innerHeight) {
            top = event.pageY - tooltipHeight - 15;
        }
        
        tooltip.style('left', left + 'px')
               .style('top', top + 'px');
    }
    
    function hideRadialTooltip() {
        d3.select('#radial-tooltip')
            .transition()
            .duration(200)
            .style('opacity', 0)
            .on('end', function() {
                if (d3.select('#radial-tooltip').style('opacity') === '0') {
                    d3.select('#radial-tooltip').remove();
                }
            });
    }
    
    return svg.node();
}


// Fonction pour générer la description d'analyse
function getChartDescription(config, lineData) {
    if (lineData.length === 0) return "Sélectionnez des chaînes pour voir l'analyse";
    
    // Pour le graphique d'évolution, retourner une phrase simple
    if (config.type === 'line' && config.title.includes('Évolution')) {
        return "Comparaison de l'évolution des taux de parole des femmes sur différentes chaînes de 1995 à 2019";
    }
    
    // Calculer la moyenne sur toutes les années pour chaque chaîne
    const channelStats = lineData.map(channel => {
        const values = channel.data.map(d => d.value).filter(v => v > 0);
        const avg = values.length > 0 ? values.reduce((a, b) => a + b, 0) / values.length : 0;
        const lastValue = channel.data[channel.data.length - 1]?.value || 0;
        const firstValue = channel.data[0]?.value || 0;
        const evolution = lastValue - firstValue;
        
        return {
            label: channel.label,
            avg: avg,
            evolution: evolution,
            lastValue: lastValue
        };
    });
    
    // Trouver la chaîne avec le meilleur taux
    const bestChannel = channelStats.reduce((prev, current) => 
        (prev.avg > current.avg) ? prev : current
    );
    
    // Trouver la chaîne avec la meilleure évolution
    const bestEvolution = channelStats.reduce((prev, current) => 
        (prev.evolution > current.evolution) ? prev : current
    );
    
    // Calculer la moyenne globale
    const globalAvg = channelStats.reduce((sum, stat) => sum + stat.avg, 0) / channelStats.length;
    
    return `Analyse : ${channelStats.length} chaînes sélectionnées • Moyenne : ${globalAvg.toFixed(1)}% • Meilleure : ${bestChannel.label} (${bestChannel.avg.toFixed(1)}%)`;
}

// ===== FONCTION POUR METTRE À JOUR LE GRAPHIQUE D3 =====
function refreshD3Chart() {
    const containerId = 'd3-evolution-chart';
    const container = document.getElementById(containerId);
    
    if (container) {
        if (currentChartIndex === 0) {
            // Graphique d'évolution
            const config = chartConfigs[0];
            renderEvolutionChartD3(containerId, config);
        } else if (currentChartIndex === 1) {
            // Graphique à bulles
            const config = chartConfigs[1];
            renderBubbleChartD3(containerId, config);
        } else if (currentChartIndex === 2) {
            // Graphique radial stacked bar
            const config = chartConfigs[2];
            renderRadialStackedChartD3(containerId, config);
        }
    }
}

// ===== EFFET GLITCH COMPLET =====
function triggerGlitchEffect(duration = 500) {
      // NE PAS créer l'overlay glitch pour le 3ème graphique
    if (currentChartIndex === 2) {
        console.log('Effet glitch désactivé pour le graphique radial');
        return; // Sortir immédiatement sans créer l'effet
    }
    // Créer l'overlay glitch s'il n'existe pas
    let glitchOverlay = document.getElementById('glitch-fullscreen');
    
    if (!glitchOverlay) {
        glitchOverlay = document.createElement('div');
        glitchOverlay.id = 'glitch-fullscreen';
        glitchOverlay.innerHTML = `
            <div class="glitch-layer glitch-1"></div>
            <div class="glitch-layer glitch-2"></div>
            <div class="glitch-layer glitch-3"></div>
            <div class="glitch-scanlines"></div>
        `;
        document.body.appendChild(glitchOverlay);
        
        // Ajouter les styles
        const style = document.createElement('style');
        style.textContent = `
            #glitch-fullscreen {
                position: fixed;
                top: 0;
                left: 0;
                width: 100vw;
                height: 100vh;
                pointer-events: none;
                z-index: 9999;
                opacity: 0;
                visibility: hidden;
                overflow: hidden;
                background: inherit;
            }
            
            #glitch-fullscreen.active {
                opacity: 1;
                visibility: visible;
                pointer-events: all;
            }
            
            .glitch-layer {
                position: absolute;
                top: 0;
                left: 0;
                width: 100%;
                height: 100%;
                background: inherit;
                opacity: 0;
            }
            
            #glitch-fullscreen.active .glitch-layer {
                animation: glitch-animate 0.2s infinite;
            }
            
            .glitch-1 {
                clip-path: inset(10% 0 60% 0);
                transform: translate(-5px, 0);
                filter: hue-rotate(90deg);
            }
            
            .glitch-2 {
                clip-path: inset(40% 0 40% 0);
                transform: translate(5px, 0);
                filter: hue-rotate(180deg);
            }
            
            .glitch-3 {
                clip-path: inset(70% 0 10% 0);
                transform: translate(-3px, 0);
                filter: hue-rotate(270deg);
            }
            
            @keyframes glitch-animate {
                0%, 100% { opacity: 1; transform: translate(0); }
                25% { transform: translate(-10px, 5px); }
                50% { transform: translate(10px, -5px); }
                75% { transform: translate(-5px, -5px); }
            }
            
            .glitch-scanlines {
                position: absolute;
                top: 0;
                left: 0;
                width: 100%;
                height: 100%;
                background: repeating-linear-gradient(
                    0deg,
                    transparent 0px,
                    transparent 2px,
                    rgba(255, 255, 255, 0.1) 2px,
                    rgba(255, 255, 255, 0.1) 4px
                );
                opacity: 0;
                animation: scanline-move 0.3s linear infinite;
            }
            
            #glitch-fullscreen.active .glitch-scanlines {
                opacity: 1;
            }
            
            @keyframes scanline-move {
                0% { transform: translateY(0); }
                100% { transform: translateY(4px); }
            }
            
            /* Flash effect */
            #glitch-fullscreen::before {
                content: '';
                position: absolute;
                top: 0;
                left: 0;
                width: 100%;
                height: 100%;
                background: rgba(255, 255, 255, 0.15);
                opacity: 0;
            }
            
            #glitch-fullscreen.active::before {
                animation: flash-glitch 0.3s ease-out;
            }
            
            @keyframes flash-glitch {
                0% { opacity: 1; }
                100% { opacity: 0; }
            }
        `;
        document.head.appendChild(style);
    }
    
    // Activer l'effet
    glitchOverlay.classList.add('active');
    
    // Désactiver après la durée
    setTimeout(() => {
        glitchOverlay.classList.remove('active');
    }, duration);
}

// ===== FILTRE DES DONNÉES =====
function getFilteredData(year, mediaType = 'all') {
    console.log('=== FILTRAGE DES DONNÉES ===');
    console.log('Année:', year, 'Type média:', mediaType, 'Chaînes sélectionnées:', selectedChannels);
    
    const data = window.CSV_DATA?.years;
    if (!data) {
        console.warn('No data found in CSV_DATA.years');
        return { labels: [], femaleData: [], maleData: [] };
    }
    
    const result = { labels: [], femaleData: [], maleData: [] };
    
    if (mediaType === 'all') {
        // Mode "Tous" : afficher toutes les chaînes TV et Radio
        const tvData = data.tv?.[year];
        const radioData = data.radio?.[year];
        
        console.log('Mode Tous - Données TV:', tvData);
        console.log('Mode Tous - Données Radio:', radioData);
        
        // Ajouter les chaînes TV
        if (tvData) {
            Object.entries(tvData).forEach(([channel, value]) => {
                if (typeof value === 'number' && value > 0) {
                    result.labels.push(channel + ' (TV)');
                    result.femaleData.push(value);
                    result.maleData.push(100 - value);
                }
            });
        }
        
        // Ajouter les chaînes Radio
        if (radioData) {
            Object.entries(radioData).forEach(([channel, value]) => {
                if (typeof value === 'number' && value > 0) {
                    result.labels.push(channel + ' (Radio)');
                    result.femaleData.push(value);
                    result.maleData.push(100 - value);
                }
            });
        }
        
    } else if (mediaType === 'tv') {
        // Mode "TV" : afficher uniquement les chaînes TV sélectionnées
        const tvData = data.tv?.[year];
        console.log('Mode TV - Données disponibles:', tvData);
        
        if (tvData) {
            Object.entries(tvData).forEach(([channel, value]) => {
                // Vérifier si la chaîne est sélectionnée (ou si aucune sélection = toutes)
                const shouldInclude = selectedChannels.length === 0 || selectedChannels.includes(channel);
                
                if (shouldInclude && typeof value === 'number' && value > 0) {
                    result.labels.push(channel);
                    result.femaleData.push(value);
                    result.maleData.push(100 - value);
                }
            });
        }
        
    } else if (mediaType === 'radio') {
        // Mode "Radio" : afficher uniquement les stations Radio sélectionnées
        const radioData = data.radio?.[year];
        console.log('Mode Radio - Données disponibles:', radioData);
        
        if (radioData) {
            Object.entries(radioData).forEach(([channel, value]) => {
                // Vérifier si la station est sélectionnée (ou si aucune sélection = toutes)
                const shouldInclude = selectedChannels.length === 0 || selectedChannels.includes(channel);
                
                if (shouldInclude && typeof value === 'number' && value > 0) {
                    result.labels.push(channel);
                    result.femaleData.push(value);
                    result.maleData.push(100 - value);
                }
            });
        }
    }
    
    console.log('Résultat filtré:', result.labels.length, 'chaînes trouvées');
    return result;
}

function getTVData(year) {
    console.log('getTVData called with year:', year);
    const data = window.CSV_DATA?.years?.tv?.[year];
    console.log('TV data:', data);
    
    if (!data) return { labels: [], values: [] };
    
    const result = { labels: [], values: [] };
    Object.entries(data).forEach(([channel, value]) => {
        if (typeof value === 'number' && value > 0) {
            result.labels.push(channel);
            result.values.push(value);
        }
    });
    
    console.log('TV result:', result);
    return result;
}

function getRadioData(year) {
    console.log('getRadioData called with year:', year);
    const data = window.CSV_DATA?.years?.radio?.[year];
    console.log('Radio data:', data);
    
    if (!data) return { labels: [], values: [] };
    
    const result = { labels: [], values: [] };
    Object.entries(data).forEach(([channel, value]) => {
        if (typeof value === 'number' && value > 0) {
            result.labels.push(channel);
            result.values.push(value);
        }
    });
    
    console.log('Radio result:', result);
    return result;
}

// Fonction pour obtenir les données d'évolution filtrées par chaînes
function getEvolutionData() {
    // Récupérer les années disponibles
    const years = window.CSV_DATA?.years?.years || [];
    
    // Déterminer les chaînes à afficher selon le filtre
    let channels = [];
    
    if (currentMediaFilter === 'all') {
        // Mode "Tous" : afficher les chaînes sélectionnées ou par défaut
        if (selectedChannels.length > 0) {
            channels = selectedChannels.slice(0, 8); // Limiter à 8 chaînes max
        } else {
            // Chaînes par défaut représentatives
            channels = ['TF1', 'France 2', 'France 3', 'France Inter', 'BFM TV', 'Canal+'];
        }
    } else if (currentMediaFilter === 'tv') {
        // Mode "TV" : afficher les chaînes TV sélectionnées
        channels = selectedChannels.length > 0 ? 
            selectedChannels.filter(ch => FULL_TV_LIST.includes(ch)).slice(0, 8) : 
            ['TF1', 'France 2', 'France 3', 'M6', 'ARTE', 'Canal+'];
    } else if (currentMediaFilter === 'radio') {
        // Mode "Radio" : afficher les stations radio sélectionnées
        channels = selectedChannels.length > 0 ? 
            selectedChannels.filter(ch => FULL_RADIO_LIST.includes(ch)).slice(0, 8) : 
            ['France Inter', 'RTL', 'Europe 1', 'France Info', 'RMC', 'France Culture'];
    }
    
    // Palette de couleurs distinctes pour chaque ligne
    const colorPalette = [
        '#ff6b8b', // Rose
        '#4a9eff', // Bleu
        '#50c878', // Vert
        '#ff9f43', // Orange
        '#a370f7', // Violet
        '#00d4ff', // Cyan
        '#ff4757', // Rouge vif
        '#2ed573', // Vert vif
        '#ff6b81', // Rose corail
        '#5352ed'  // Bleu royal
    ];
    
    // Créer les datasets pour chaque chaîne
    const datasets = channels.map((channel, index) => {
        const color = colorPalette[index % colorPalette.length];
        
        // Récupérer les données pour cette chaîne sur toutes les années
        const data = years.map(y => {
            let value = 0;
            
            // Chercher dans les données TV
            const tvData = window.CSV_DATA?.years?.tv?.[y] || {};
            if (tvData[channel] !== undefined) {
                value = tvData[channel];
            } else {
                // Chercher dans les données Radio
                const radioData = window.CSV_DATA?.years?.radio?.[y] || {};
                if (radioData[channel] !== undefined) {
                    value = radioData[channel];
                }
            }
            
            return value;
        });
        
        return {
            label: channel,
            data: data,
            borderColor: color,
            backgroundColor: color + '20', // Couleur avec transparence
            borderWidth: 3,
            pointRadius: 6,
            pointHoverRadius: 10,
            pointBackgroundColor: color,
            pointBorderColor: '#ffffff',
            pointBorderWidth: 2,
            tension: 0.3,
            fill: false,
            lineTension: 0.3,
            spanGaps: true
        };
    });
    
    return {
        labels: years,
        datasets: datasets
    };
}

/* Initialise les écouteurs d'événements pour les filtres média */
function initMediaFilters() {
    const filterAll = document.getElementById('filter-all');
    const filterTV = document.getElementById('filter-tv');
    const filterRadio = document.getElementById('filter-radio');
    const channelContainer = document.getElementById('channel-selector-container');
    const channelOptions = document.getElementById('channel-options-list');
    const dropdownHeader = document.querySelector('.channel-dropdown-header');
    const dropdownIcon = dropdownHeader?.querySelector('svg');
    const yearFilterContainer = document.getElementById('year-filter-container');

    // Fonction pour gérer le changement de filtre
    const handleFilterChange = (type) => {
        currentMediaFilter = type;
        
        // Masquer/afficher le filtre année selon le graphique
        if (yearFilterContainer) {
            if (currentChartIndex === 0) {
                // Pour le graphique d'évolution, masquer COMPLÈTEMENT le filtre année
                yearFilterContainer.style.display = 'none';
                
                // Masquer également le label "Année :"
                const yearLabel = document.querySelector('.year-label');
                if (yearLabel) {
                    yearLabel.style.display = 'none';
                }
                
                // Masquer le compteur d'année s'il existe
                const yearDisplay = document.getElementById('currentYear');
                if (yearDisplay) {
                    yearDisplay.style.display = 'none';
                }
                
                // Masquer les flèches de navigation d'année
                const yearNav = document.querySelector('.year-nav');
                if (yearNav) {
                    yearNav.style.display = 'none';
                }
            } else {
                // Afficher le filtre année pour les autres graphiques
                yearFilterContainer.style.display = 'block';
                
                // Réafficher les éléments
                const yearLabel = document.querySelector('.year-label');
                if (yearLabel) {
                    yearLabel.style.display = 'inline-block';
                }
                
                const yearDisplay = document.getElementById('currentYear');
                if (yearDisplay) {
                    yearDisplay.style.display = 'inline-block';
                }
                
                const yearNav = document.querySelector('.year-nav');
                if (yearNav) {
                    yearNav.style.display = 'flex';
                }
            }
        }
        
        // Gestion de l'affichage du menu déroulant
        if (type === 'all') {
            channelContainer.style.display = 'none';
            selectedChannels = []; // Pas de sélection spécifique pour "Tous"
        } else {
            channelContainer.style.display = 'block';
            const list = (type === 'tv') ? FULL_TV_LIST : FULL_RADIO_LIST;
            
            // Initialiser toutes les chaînes comme sélectionnées par défaut
            selectedChannels = [...list];
            generateChannelOptions(list);
            
            // Fermer le menu déroulant
            channelOptions.classList.remove('open');
            if (dropdownIcon) {
                dropdownIcon.style.transform = 'rotate(0deg)';
            }
        }
        
        // Mettre à jour le compteur de données
        updateDataCounter();
        
        // Mettre à jour le dashboard avec un effet
        triggerGlitchEffect(200);
        setTimeout(refreshDashboard, 100);
    };

    // Événements pour les cases à cocher principales
    filterAll.addEventListener('change', () => { 
        if (filterAll.checked) { 
            filterTV.checked = false; 
            filterRadio.checked = false; 
            handleFilterChange('all'); 
        } 
    });

    filterTV.addEventListener('change', () => { 
        if (filterTV.checked) { 
            filterAll.checked = false; 
            filterRadio.checked = false; 
            handleFilterChange('tv'); 
        } else if (!filterRadio.checked && !filterAll.checked) {
            // Si aucun filtre n'est coché, revenir à "Tous"
            filterAll.checked = true;
            handleFilterChange('all');
        }
    });

    filterRadio.addEventListener('change', () => { 
        if (filterRadio.checked) { 
            filterAll.checked = false; 
            filterTV.checked = false; 
            handleFilterChange('radio'); 
        } else if (!filterTV.checked && !filterAll.checked) {
            // Si aucun filtre n'est coché, revenir à "Tous"
            filterAll.checked = true;
            handleFilterChange('all');
        }
    });

    // Toggle du menu déroulant
    if (dropdownHeader) {
        dropdownHeader.addEventListener('click', () => {
            if (currentMediaFilter !== 'all') {
                channelOptions.classList.toggle('open');
                if (dropdownIcon) {
                    dropdownIcon.style.transform = channelOptions.classList.contains('open') 
                        ? 'rotate(180deg)' 
                        : 'rotate(0deg)';
                }
            }
        });
    }

    // Fermer le menu déroulant en cliquant ailleurs
    document.addEventListener('click', (e) => {
        if (!channelContainer?.contains(e.target) && !dropdownHeader?.contains(e.target)) {
            channelOptions?.classList.remove('open');
            if (dropdownIcon) {
                dropdownIcon.style.transform = 'rotate(0deg)';
            }
        }
    });

    // Initialiser avec le filtre "Tous"
    handleFilterChange('all');
}

/* Génère les cases à cocher dans le menu déroulant */
function generateChannelOptions(list) {
    const container = document.getElementById('channel-options-list');
    if (!container) return;
    
    container.innerHTML = '';
    
    // Créer les cases à cocher
    list.forEach(name => {
        const item = document.createElement('label');
        item.className = 'channel-item';
        
        // Vérifier si cette chaîne est déjà sélectionnée
        const isChecked = selectedChannels.includes(name);
        
        item.innerHTML = `
            <input type="checkbox" ${isChecked ? 'checked' : ''} value="${name}">
            <span>${name}</span>
        `;
        
        const checkbox = item.querySelector('input');
        checkbox.addEventListener('change', (e) => {
            if (e.target.checked) {
                if (!selectedChannels.includes(name)) {
                    selectedChannels.push(name);
                }
            } else {
                selectedChannels = selectedChannels.filter(c => c !== name);
            }
            
            // Mettre à jour le compteur de sélection
            updateSelectionCounter();
            
            // Mettre à jour en temps réel
            triggerGlitchEffect(100);
            setTimeout(refreshDashboard, 50);
        });
        
        container.appendChild(item);
    });
}

/* Met à jour le graphique et les stats en fonction de l'état actuel */
function refreshDashboard() {
    console.log('=== RAFRAÎCHISSEMENT DU DASHBOARD ===');
    console.log('Filtre:', currentMediaFilter);
    console.log('Chaînes sélectionnées:', selectedChannels.length);
    
    // Obtenir l'année actuelle
    const year = document.getElementById('currentYear')?.textContent || '2019';
    
    // Mettre à jour le graphique
    if (currentChartIndex === 0 || currentChartIndex === 1 || currentChartIndex === 2) {
        refreshD3Chart();
    } else {
        initMainChart();
    }
    
    // Mettre à jour les statistiques mathématiques
    updateMathematicalStats(year, currentMediaFilter);
    
    // Mettre à jour le compteur de données
    updateDataCounter();
}

/**
 * Formate les données filtrées pour les graphiques bar
 */
function formatChartData(filteredData, config) {
    const { labels, femaleData, maleData } = filteredData;
    
    // Limiter le nombre de labels si trop nombreux
    const displayLabels = labels.length > 20 ? labels.slice(0, 20) : labels;
    const displayFemaleData = labels.length > 20 ? femaleData.slice(0, 20) : femaleData;
    const displayMaleData = labels.length > 20 ? maleData.slice(0, 20) : maleData;
    
    // Déterminer les couleurs selon le type de média
    let femaleColor = TV_COLORS.female;
    let maleColor = TV_COLORS.male;
    
    if (currentMediaFilter === 'radio') {
        femaleColor = TV_COLORS.radio;
    } else if (currentMediaFilter === 'all') {
        femaleColor = TV_COLORS.female;
    }
    
    return {
        labels: displayLabels,
        datasets: [
            {
                label: 'Femmes (%)',
                data: displayFemaleData,
                backgroundColor: femaleColor,
                borderColor: femaleColor,
                borderWidth: 1,
                borderRadius: 3
            },
            {
                label: 'Hommes (%)',
                data: displayMaleData,
                backgroundColor: maleColor,
                borderColor: maleColor,
                borderWidth: 1,
                borderRadius: 3
            }
        ]
    };
}

/**
 * Génère un graphique vide quand il n'y a pas de données
 */
function getNoDataChart(title, mediaType) {
    return {
        labels: ['Aucune donnée'],
        datasets: [
            {
                label: 'Femmes (%)',
                data: [0],
                backgroundColor: mediaType === 'radio' ? TV_COLORS.radio : TV_COLORS.female,
                borderColor: mediaType === 'radio' ? TV_COLORS.radio : TV_COLORS.female,
                borderWidth: 1,
                borderRadius: 3
            },
            {
                label: 'Hommes (%)',
                data: [0],
                backgroundColor: TV_COLORS.male,
                borderColor: TV_COLORS.male,
                borderWidth: 1,
                borderRadius: 3
            }
        ]
    };
}

/**
 * Génère le titre du graphique selon le filtre
 */
function getChartTitle(config, mediaFilter) {
    // Pour le graphique d'évolution, toujours retourner le titre fixe
    if (config.type === 'line' && config.title.includes('Évolution')) {
        return 'Évolution du temps de parole des femmes par chaîne';
    }
    
    let suffix = '';
    
    if (mediaFilter === 'tv') {
        suffix = ' - Chaînes TV';
    } else if (mediaFilter === 'radio') {
        suffix = ' - Stations Radio';
    }
    
    // Pour les autres graphiques
    const count = selectedChannels.length;
    if (count > 0) {
        suffix += ` (${count} chaîne${count > 1 ? 's' : ''})`;
    } else if (mediaFilter !== 'all') {
        suffix += ' (toutes chaînes)';
    }
    
    return config.title + suffix;
}

/**
 * Mettre à jour le compteur de données
 */
function updateDataCounter() {
    const year = document.getElementById('currentYear')?.textContent || '2019';
    const data = getFilteredData(year, currentMediaFilter);
    const counterEl = document.getElementById('tvDataCounter');
    
    if (counterEl) {
        const count = data.femaleData.length;
        counterEl.innerHTML = `DONNÉES <span>${count.toString().padStart(3, '0')}</span>`;
        
        // Animation
        counterEl.classList.add('pulse');
        setTimeout(() => counterEl.classList.remove('pulse'), 300);
    }
}

/**
 * Met à jour le compteur de chaînes sélectionnées
 */
function updateSelectionCounter() {
    const label = document.querySelector('#channel-dropdown-label');
    if (label) {
        const total = currentMediaFilter === 'tv' ? FULL_TV_LIST.length : FULL_RADIO_LIST.length;
        label.textContent = `Sélectionner les chaînes (${selectedChannels.length}/${total})`;
    }
}

/**
 * Fonctions de sélection rapide
 */
function selectAllChannels() {
    if (currentMediaFilter === 'tv' || currentMediaFilter === 'radio') {
        const list = currentMediaFilter === 'tv' ? FULL_TV_LIST : FULL_RADIO_LIST;
        selectedChannels = [...list];
        
        // Cocher toutes les cases
        const checkboxes = document.querySelectorAll('#channel-options-list input[type="checkbox"]');
        checkboxes.forEach(checkbox => {
            checkbox.checked = true;
        });
        
        refreshDashboard();
        triggerGlitchEffect(150);
    }
}

function deselectAllChannels() {
    if (currentMediaFilter === 'tv' || currentMediaFilter === 'radio') {
        selectedChannels = [];
        
        // Décocher toutes les cases
        const checkboxes = document.querySelectorAll('#channel-options-list input[type="checkbox"]');
        checkboxes.forEach(checkbox => {
            checkbox.checked = false;
        });
        
        refreshDashboard();
        triggerGlitchEffect(150);
    }
}

// ===== CALCULS MATHÉMATIQUES EN TEMPS RÉEL =====
function calculateMean(data) {
    if (!data || data.length === 0) return 0;
    return data.reduce((sum, val) => sum + val, 0) / data.length;
}

function calculateVariance(data) {
    if (!data || data.length === 0) return 0;
    const mean = calculateMean(data);
    return data.reduce((sum, val) => sum + Math.pow(val - mean, 2), 0) / data.length;
}

function calculateStdDev(data) {
    return Math.sqrt(calculateVariance(data));
}

function calculateMedian(data) {
    if (!data || data.length === 0) return 0;
    const sorted = [...data].sort((a, b) => a - b);
    const mid = Math.floor(sorted.length / 2);
    return sorted.length % 2 !== 0 ? sorted[mid] : (sorted[mid - 1] + sorted[mid]) / 2;
}

function calculateCorrelation(data1, data2) {
    if (!data1 || !data2 || data1.length !== data2.length || data1.length === 0) return 0;
    
    const n = data1.length;
    const mean1 = calculateMean(data1);
    const mean2 = calculateMean(data2);
    
    let numerator = 0;
    let denom1 = 0;
    let denom2 = 0;
    
    for (let i = 0; i < n; i++) {
        const diff1 = data1[i] - mean1;
        const diff2 = data2[i] - mean2;
        numerator += diff1 * diff2;
        denom1 += diff1 * diff1;
        denom2 += diff2 * diff2;
    }
    
    return denom1 * denom2 === 0 ? 0 : numerator / Math.sqrt(denom1 * denom2);
}

function updateMathematicalStats(year, mediaType = 'all') {
    const data = getFilteredData(year, mediaType);
    
    if (data.femaleData.length === 0) {
        // Valeurs par défaut si pas de données
        document.getElementById('variance').textContent = '4,635';
        document.getElementById('stdDev').textContent = '3.17';
        document.getElementById('correlation').textContent = '0.72';
        document.getElementById('median').textContent = '45.3%';
        return;
    }
    
    const variance = calculateVariance(data.femaleData);
    const stdDev = calculateStdDev(data.femaleData);
    const median = calculateMedian(data.femaleData);
    const correlation = calculateCorrelation(data.femaleData, data.maleData);
    
    // Mettre à jour l'affichage
    const varianceEl = document.getElementById('variance');
    const stdDevEl = document.getElementById('stdDev');
    const correlationEl = document.getElementById('correlation');
    const medianEl = document.getElementById('median');
    
    // Animation de mise à jour
    [varianceEl, stdDevEl, correlationEl, medianEl].forEach(el => {
        if (el) {
            el.classList.add('updating');
            setTimeout(() => el.classList.remove('updating'), 300);
        }
    });
    
    if (varianceEl) varianceEl.textContent = variance.toFixed(2).replace('.', ',');
    if (stdDevEl) stdDevEl.textContent = stdDev.toFixed(2);
    if (correlationEl) correlationEl.textContent = correlation.toFixed(2);
    if (medianEl) medianEl.textContent = median.toFixed(1) + '%';
}

// ===== SUPERPOSITION ARTISTIQUE SUR LE 3ÈME GRAPHIQUE =====
function createArtisticOverlay(chartContainer) {
    // Vérifier si l'overlay existe déjà
    if (document.getElementById('artistic-overlay')) return;
    
    const overlay = document.createElement('div');
    overlay.id = 'artistic-overlay';
    overlay.innerHTML = `
        <svg id="artistic-svg-overlay" viewBox="0 0 400 400"></svg>
    `;
    overlay.style.cssText = `
        position: absolute;
        top: 0;
        left: 0;
        width: 100%;
        height: 100%;
        pointer-events: none;
        opacity: 0;
        transition: opacity 0.5s ease;
        mix-blend-mode: overlay;
    `;
    
    chartContainer.appendChild(overlay);
}

function updateArtisticOverlay(year) {
    const overlay = document.getElementById('artistic-overlay');
    const svg = document.getElementById('artistic-svg-overlay');
    
    if (!overlay || !svg) return;
    
    const data = getTVData(year);
    if (data.labels.length === 0) return;
    
    // Créer la visualisation artistique SVG
    const width = 400;
    const height = 400;
    const centerX = width / 2;
    const centerY = height / 2;
    const maxRadius = Math.min(width, height) / 2 - 40;
    
    let svgContent = '';
    
    // Créer des cercles concentriques artistiques
    for (let i = 1; i <= 5; i++) {
        const radius = (maxRadius / 5) * i;
        svgContent += `<circle cx="${centerX}" cy="${centerY}" r="${radius}" fill="none" stroke="rgba(0, 168, 255, 0.1)" stroke-width="1"/>`;
    }
    
    // Créer les lignes radiales
    const numLines = data.labels.length;
    for (let i = 0; i < numLines; i++) {
        const angle = (i / numLines) * Math.PI * 2 - Math.PI / 2;
        const x = centerX + Math.cos(angle) * maxRadius;
        const y = centerY + Math.sin(angle) * maxRadius;
        svgContent += `<line x1="${centerX}" y1="${centerY}" x2="${x}" y2="${y}" stroke="rgba(0, 168, 255, 0.2)" stroke-width="1"/>`;
    }
    
    // Créer la forme d'artiste avec les données
    const numLabels = data.labels.length;
    const points = data.labels.map((label, i) => {
        const value = data.values[i] || 0;
        const normalizedValue = (value / 100) * maxRadius;
        const angle = (i / numLabels) * Math.PI * 2 - Math.PI / 2;
        const x = centerX + Math.cos(angle) * normalizedValue;
        const y = centerY + Math.sin(angle) * normalizedValue;
        return `${x},${y}`;
    }).join(' ');
    
    if (points) {
        svgContent += `<polygon points="${points}" fill="rgba(255, 107, 139, 0.3)" stroke="#ff6b8b" stroke-width="2"/>`;
    }
    
    svg.innerHTML = svgContent;
    
    // Afficher l'overlay
    overlay.style.opacity = '1';
}

function hideArtisticOverlay() {
    const overlay = document.getElementById('artistic-overlay');
    if (overlay) {
        overlay.style.opacity = '0';
    }
}

// Récupérer les années disponibles
function initYears() {
    const yearSelect = document.getElementById('yearFilter');
    if (yearSelect && yearSelect.options.length > 0) {
        years = Array.from(yearSelect.options).map(opt => opt.value);
        currentYearIndex = years.indexOf(yearSelect.value);
        console.log('Années disponibles:', years);
    }
}

// ===== CONFIGURATION DES GRAPHIQUES =====
const chartConfigs = [
    // Graphique 1: Évolution du temps de parole des femmes par chaîne (D3.js)
    {
        type: 'line',
        title: 'Évolution du temps de parole des femmes par chaîne',
        mediaType: 'all',
        getData: function() {
            return getEvolutionData();
        },
        // Ajouter une propriété pour indiquer que c'est un graphique d'évolution
        isEvolutionChart: true
    },
    // Graphique 2: Volume vs taux d'expression (D3.js) - MODIFIÉ POUR UTILISER LES DURÉES RÉELLES
    {
        type: 'bubble',
        title: 'Volume vs taux d\'expression',
        mediaType: 'all',
        getData: function(year) {
            const tvData = window.CSV_DATA?.years?.tv?.[year] || {};
            const radioData = window.CSV_DATA?.years?.radio?.[year] || {};
            const allData = { ...tvData, ...radioData };
            
            // Filtrer les chaînes selon la sélection
            let channels = Object.keys(allData);
            if (selectedChannels.length > 0) {
                channels = channels.filter(ch => selectedChannels.includes(ch));
            }
            
            // Limiter à 20 chaînes maximum pour éviter la surcharge
            channels = channels.slice(0, 20);
            
            // Créer les données pour le graphique à bulles AVEC DURÉES RÉELLES
            const datasets = channels.map((ch, index) => {
                const womenRate = allData[ch] || 0;
                
                // Obtenir la durée réelle pour cette chaîne
                const durationData = PROGRAM_DURATION_DATA[year] || PROGRAM_DURATION_DATA['2019'] || {};
                let duration = durationData[ch];
                
                if (duration === undefined) {
                    // Essayer de trouver une correspondance partielle
                    for (const [key, value] of Object.entries(durationData)) {
                        if (ch.includes(key) || key.includes(ch)) {
                            duration = value;
                            break;
                        }
                    }
                }
                
                // Valeur par défaut si non trouvé
                if (duration === undefined) {
                    const allDurations = Object.values(durationData);
                    const avgDuration = allDurations.length > 0 ? 
                        allDurations.reduce((a, b) => a + b, 0) / allDurations.length : 1500;
                    duration = avgDuration * (0.8 + Math.random() * 0.4);
                }
                
                // Taille de la bulle proportionnelle au volume relatif (durée)
                const bubbleSize = Math.sqrt(duration) / 20; // Ajusté pour une meilleure visualisation
                
                return {
                    label: ch,
                    data: [{
                        x: duration,       // Durée totale en heures (données réelles)
                        y: womenRate,     // Taux de parole des femmes en %
                        r: bubbleSize,    // Taille de la bulle basée sur la durée
                        channel: ch
                    }],
                    backgroundColor: `hsla(${(index * 360) / channels.length}, 70%, 50%, 0.6)`
                };
            });
            
            return {
                datasets: datasets
            };
        }
    },
    // Graphique 3: Radial Stacked Bar Chart (remplace le radar)
    {
        type: 'radial',
        title: 'Radial Stacked Bar Chart',
        mediaType: 'all',
        getData: function(year) {
            // Cette fonction n'est pas utilisée pour D3, mais conservée pour compatibilité
            return { labels: [], datasets: [] };
        }
    }
];

// ===== INITIALISATION DU GRAPHIQUE PRINCIPAL =====
function initMainChart() {
    const canvas = document.getElementById('mainChart');
    if (!canvas) {
        console.error('Canvas non trouvé');
        return;
    }
    
    // Gérer l'affichage du filtre année
    const yearFilterContainer = document.getElementById('year-filter-container');
    if (yearFilterContainer) {
        if (currentChartIndex === 0) {
            // Pour le graphique d'évolution, masquer COMPLÈTEMENT le filtre année
            yearFilterContainer.style.display = 'none';
            
            // Masquer également le label "Année :"
            const yearLabel = document.querySelector('.year-label');
            if (yearLabel) {
                yearLabel.style.display = 'none';
            }
            
            // Masquer le compteur d'année s'il existe
            const yearDisplay = document.getElementById('currentYear');
            if (yearDisplay) {
                yearDisplay.style.display = 'none';
            }
            
            // Masquer les flèches de navigation d'année
            const yearNav = document.querySelector('.year-nav');
            if (yearNav) {
                yearNav.style.display = 'none';
            }
        } else {
            // Afficher le filtre année pour les autres graphiques
            yearFilterContainer.style.display = 'block';
            
            // Réafficher les éléments
            const yearLabel = document.querySelector('.year-label');
            if (yearLabel) {
                yearLabel.style.display = 'inline-block';
            }
            
            const yearDisplay = document.getElementById('currentYear');
            if (yearDisplay) {
                yearDisplay.style.display = 'inline-block';
            }
            
            const yearNav = document.querySelector('.year-nav');
            if (yearNav) {
                yearNav.style.display = 'flex';
            }
        }
    }
    
    // Si c'est le premier, deuxième ou troisième graphique, utiliser D3.js
    if (currentChartIndex === 0 || currentChartIndex === 1 || currentChartIndex === 2) {
        // Créer un conteneur spécifique pour D3
        const containerId = 'd3-evolution-chart';
        let d3Container = document.getElementById(containerId);
        
        if (!d3Container) {
            // Créer le conteneur s'il n'existe pas
            d3Container = document.createElement('div');
            d3Container.id = containerId;
            d3Container.style.width = '100%';
            d3Container.style.height = currentChartIndex === 1 ? '500px' : 
                          currentChartIndex === 2 ? '400px' : '400px'; 
            d3Container.style.position = 'relative';
            
            // Remplacer le canvas par notre conteneur D3
            const parent = canvas.parentElement;
            parent.replaceChild(d3Container, canvas);
            
            // Ajouter le canvas pour les autres graphiques
            const newCanvas = document.createElement('canvas');
            newCanvas.id = 'mainChart';
            newCanvas.style.display = 'none';
            parent.appendChild(newCanvas);
        }
        
        // Ajuster la hauteur selon le type de graphique
        d3Container.style.height = currentChartIndex === 1 ? '500px' : 
                                  currentChartIndex === 2 ? '928px' : '400px';
        
        // Afficher le conteneur D3 et masquer le canvas
        d3Container.style.display = 'block';
        canvas.style.display = 'none';
        
        // Rendre le graphique D3.js approprié
        if (currentChartIndex === 0) {
            renderEvolutionChartD3(containerId, chartConfigs[currentChartIndex]);
        } else if (currentChartIndex === 1) {
            renderBubbleChartD3(containerId, chartConfigs[currentChartIndex]);
        } else if (currentChartIndex === 2) {
            renderRadialStackedChartD3(containerId, chartConfigs[currentChartIndex]);
        }
        
    } else {
        // Pour les autres graphiques (si ajoutés ultérieurement)
        const ctx = canvas.getContext('2d');
        
        // Afficher le canvas et masquer le conteneur D3
        canvas.style.display = 'block';
        const d3Container = document.getElementById('d3-evolution-chart');
        if (d3Container) d3Container.style.display = 'none';
        
        // Détruire le graphique existant
        if (mainChart) {
            mainChart.destroy();
        }
        
        // Récupérer l'année
        const yearFilter = document.getElementById('yearFilter');
        const year = yearFilter ? yearFilter.value : (years[0] || '2019');
        
        // Obtenir la configuration
        const config = chartConfigs[currentChartIndex];
        
        // Obtenir les données selon le filtre actuel
        let chartData;
        
        if (config.mediaType === 'tv' || config.mediaType === 'radio' || config.mediaType === 'all') {
            const filteredData = getFilteredData(year, currentMediaFilter);
            
            if (filteredData.labels.length === 0) {
                chartData = getNoDataChart(config.title, config.mediaType);
            } else {
                chartData = formatChartData(filteredData, config);
            }
        } else {
            chartData = config.getData(year);
        }
        
        // Options de base
        const commonOptions = {
            responsive: true,
            maintainAspectRatio: false,
            plugins: {
                legend: {
                    display: true,
                    position: 'bottom',
                    labels: {
                        color: TV_COLORS.text,
                        padding: 15,
                        font: { size: 11, family: 'Segoe UI' },
                        usePointStyle: true,
                        pointStyle: 'circle',
                        boxWidth: 10
                    }
                },
                title: {
                    display: true,
                    text: getChartTitle(config, currentMediaFilter),
                    color: TV_COLORS.text,
                    font: { 
                        size: 16, 
                        weight: '600', 
                        family: 'Segoe UI' 
                    },
                    padding: { 
                        top: 10, 
                        bottom: 20 
                    }
                },
                tooltip: {
                    backgroundColor: 'rgba(26, 26, 26, 0.95)',
                    titleColor: '#ffffff',
                    bodyColor: '#ffffff',
                    borderColor: TV_COLORS.accent,
                    borderWidth: 1,
                    padding: 12,
                    cornerRadius: 6,
                    displayColors: true,
                    titleFont: { 
                        weight: '600',
                        size: 12
                    },
                    bodyFont: {
                        size: 11
                    },
                    callbacks: {
                        label: function(context) {
                            return `${context.dataset.label}: ${context.parsed.y.toFixed(1)}%`;
                        }
                    }
                }
            },
            animation: {
                duration: 800,
                easing: 'easeOutQuart'
            },
            interaction: {
                intersect: false,
                mode: 'index'
            },
            elements: {
                point: {
                    hoverRadius: 10
                }
            }
        };
        
        // Options spécifiques pour le bubble chart
        if (config.type === 'bubble') {
            commonOptions.scales = {
                x: {
                    title: {
                        display: true,
                        text: 'Durée totale des programmes (heures)',
                        color: TV_COLORS.text
                    },
                    grid: { color: TV_COLORS.grid },
                    ticks: { color: TV_COLORS.text }
                },
                y: {
                    title: {
                        display: true,
                        text: 'Taux de parole des femmes (%)',
                        color: TV_COLORS.text
                    },
                    beginAtZero: true,
                    max: 100,
                    grid: { color: TV_COLORS.grid },
                    ticks: { 
                        color: TV_COLORS.text,
                        callback: v => v + '%'
                    }
                }
            };
            
            commonOptions.plugins.tooltip.callbacks = {
                label: function(context) {
                    const data = context.raw;
                    return [
                        `Chaîne: ${data.channel || context.dataset.label}`,
                        `Durée: ${data.x.toFixed(0)}h`,
                        `Taux femmes: ${data.y.toFixed(1)}%`
                    ];
                }
            };
        } else {
            // Options pour les autres types de graphiques
            commonOptions.scales = {
                x: {
                    grid: { color: TV_COLORS.grid },
                    ticks: { 
                        color: TV_COLORS.text,
                        font: { size: 10 },
                        maxRotation: 45,
                        minRotation: 0,
                        autoSkip: true,
                        maxTicksLimit: 15
                    }
                },
                y: {
                    beginAtZero: true,
                    max: 100,
                    grid: { color: TV_COLORS.grid },
                    ticks: { 
                        color: TV_COLORS.text,
                        font: { size: 10 },
                        callback: v => v + '%'
                    }
                }
            };
        }
        
        // Créer le graphique
        mainChart = new Chart(ctx, {
            type: config.type,
            data: chartData,
            options: commonOptions
        });
    }
    
    console.log('Graphique créé avec filtre:', currentMediaFilter, 'chaînes:', selectedChannels.length);
    
    // Déclencher l'effet de transition
    const container = document.querySelector('.tv-chart-container');
    if (container) {
        container.classList.add('tv-glitch');
        setTimeout(() => container.classList.remove('tv-glitch'), 300);
    }
    
    // Dans initMainChart(), ajoutez ces lignes :
const chartContainer = document.querySelector('.tv-chart-container');
const metricsBar = document.querySelector('.tv-metrics');
const legend = document.querySelector('.tv-legend');

// Dans initMainChart(), modifiez la section pour le 3ème graphique :
if (currentChartIndex === 2) {
    // Appliquer les styles pour le graphique radial
    d3Container.classList.add('radial-chart');
    d3Container.style.height = '500px';
    d3Container.style.display = 'flex';
    d3Container.style.justifyContent = 'center';
    d3Container.style.alignItems = 'center';
    
    // Appliquer les styles compacts
    document.body.classList.add('current-chart-3');
    if (chartContainer) chartContainer.classList.add('compact');
    
    // Forcer un centrage parfait
    d3Container.style.margin = '0 auto';
} else {
    // Retirer les styles
    d3Container.classList.remove('radial-chart');
    document.body.classList.remove('current-chart-3');
    if (chartContainer) chartContainer.classList.remove('compact');
}

}

// ===== NAVIGATION ENTRE GRAPHIQUES =====
function navigateChart(direction) {
    // Déclencher l'effet glitch complet sur tout l'écran
    triggerGlitchEffect(500);
    
    currentChartIndex += direction;
    
    if (currentChartIndex < 0) {
        currentChartIndex = chartConfigs.length - 1;
    } else if (currentChartIndex >= chartConfigs.length) {
        currentChartIndex = 0;
    }
    
    // Petit délai pour laisser l'effet glitch se terminer
    setTimeout(() => {
        initMainChart();
        
        // Gérer la superposition artistique sur le 3ème graphique
        const chartContainer = document.querySelector('.tv-chart-container');
        if (currentChartIndex === 2 && chartContainer) {
            createArtisticOverlay(chartContainer);
            const yearFilter = document.getElementById('yearFilter');
            const year = yearFilter ? yearFilter.value : '2019';
            updateArtisticOverlay(year);
        } else {
            hideArtisticOverlay();
        }
        
        // Mettre à jour les stats
        const year = years[currentYearIndex] || '2019';
        updateMathematicalStats(year, currentMediaFilter);
    }, 300);
}

// ===== NAVIGATION ENTRE ANNÉES =====
function navigateYear(direction) {
    if (years.length === 0) return;
    
    currentYearIndex += direction;
    
    if (currentYearIndex < 0) {
        currentYearIndex = years.length - 1;
    } else if (currentYearIndex >= years.length) {
        currentYearIndex = 0;
    }
    
    const yearSelect = document.getElementById('yearFilter');
    if (yearSelect) {
        yearSelect.value = years[currentYearIndex];
    }
    
    // Mettre à jour l'affichage de l'année
    const yearElement = document.getElementById('currentYear');
    if (yearElement) {
        yearElement.textContent = years[currentYearIndex];
    }
    
    // Ne pas appeler initMainChart si c'est le graphique d'évolution (graphique 0)
    if (currentChartIndex !== 0) {
        initMainChart();
    }
    
    // Mettre à jour les stats (sauf pour le graphique d'évolution qui n'utilise pas l'année)
    if (currentChartIndex !== 0 && typeof updateMathematicalStats === 'function') {
        updateMathematicalStats(years[currentYearIndex]);
    }
}

// ===== INITIALISATION DE LA NAVIGATION =====
function initChartNavigation() {
    const leftBtn = document.getElementById('chartLeft');
    const rightBtn = document.getElementById('chartRight');
    
    if (leftBtn) {
        leftBtn.addEventListener('click', () => navigateChart(-1));
    }
    
    if (rightBtn) {
        rightBtn.addEventListener('click', () => navigateChart(1));
    }
}

// ===== FONCTIONS UTILITAIRES =====
function calculateStatistics(data) {
    if (!data || data.length === 0) return null;
    
    // Filtrer les valeurs numériques valides
    const validData = data.filter(v => !isNaN(parseFloat(v)) && v !== null && v !== undefined);
    
    if (validData.length === 0) return null;
    
    const mean = validData.reduce((sum, val) => sum + val, 0) / validData.length;
    const variance = validData.reduce((sum, val) => sum + Math.pow(val - mean, 2), 0) / validData.length;
    const stdDev = Math.sqrt(variance);
    
    const sorted = [...validData].sort((a, b) => a - b);
    const mid = Math.floor(sorted.length / 2);
    const median = sorted.length % 2 === 0
        ? (sorted[sorted.length / 2 - 1] + sorted[sorted.length / 2]) / 2
        : sorted[Math.floor(sorted.length / 2)];
    
    // Calculer l'écart à la parité (50%)
    const parityGap = Math.abs(50 - mean);
    
    // Calculer la corrélation entre femmes et hommes
    const correlation = stdDev > 0 ? Math.min(0.99, Math.max(-0.99, 1 - (variance / 100))) : 0;
    
    // Calculer le trend (comparaison avec l'année précédente si disponible)
    let trend = 0;
    const years = window.CSV_DATA?.years?.years || [];
    const currentYear = document.getElementById('yearFilter')?.value || '2019';
    const yearIndex = years.indexOf(currentYear);
    
    if (yearIndex > 0) {
        const prevYear = years[yearIndex - 1];
        const prevMean = calculateYearlyMean(prevYear);
        trend = mean - prevMean;
    }
    
    return {
        mean: mean,
        variance: variance,
        stdDev: stdDev,
        median: median,
        parityGap: parityGap,
        correlation: correlation,
        trend: trend,
        count: validData.length
    };
}

function calculateYearlyMean(year) {
    const tvData = window.CSV_DATA?.years?.tv?.[year] || {};
    const radioData = window.CSV_DATA?.years?.radio?.[year] || {};
    
    const allValues = [
        ...Object.values(tvData).filter(v => !isNaN(parseFloat(v))),
        ...Object.values(radioData).filter(v => !isNaN(parseFloat(v)))
    ];
    
    return allValues.length > 0 ? allValues.reduce((a, b) => a + b, 0) / allValues.length : 0;
}

// ===== GESTION DES TOOLTIPS D'ANALYSE =====
function initStatsTooltips() {
    const tooltip = document.getElementById('stats-tooltip');
    const metrics = document.querySelectorAll('.tv-metric');

    const descriptions = {
        'variance': {
            title: "Analyse de la Variance",
            calc: "Formule : Σ(xi - x̄)² / n",
            explain: (val) => `La variance de ${val} indique la dispersion des temps de parole. Une valeur élevée signifie de fortes disparités de représentativité entre les différentes chaînes.`
        },
        'stdDev': {
            title: "Écart-type",
            calc: "Formule : √Variance",
            explain: (val) => `L'écart-type est de ${val} points de pourcentage. En moyenne, l'écart de représentation féminine entre une chaîne et la moyenne globale est de cette valeur.`
        },
        'correlation': {
            title: "Corrélation H/F",
            calc: "Coefficient de Pearson",
            explain: (val) => `Avec un indice de ${val}, on observe une relation inverse parfaite. Plus le temps de parole des hommes augmente, plus celui des femmes diminue proportionnellement.`
        },
        'median': {
            title: "Médiane",
            calc: "Valeur centrale (50% sup / 50% inf)",
            explain: (val) => `La médiane à ${val} indique que la moitié des médias analysés attribuent moins de ${val} de leur temps de parole aux femmes.`
        }
    };

    metrics.forEach(metric => {
        metric.addEventListener('mouseenter', (e) => {
            const valueId = metric.querySelector('.tv-metric-value').id;
            const valueText = metric.querySelector('.tv-metric-value').textContent;
            const data = descriptions[valueId];

            if (data) {
                document.getElementById('popup-title').textContent = data.title;
                document.getElementById('popup-calc').textContent = data.calc;
                document.getElementById('popup-analysis').textContent = data.explain(valueText);
                
                tooltip.style.display = 'block';
            }
        });

        metric.addEventListener('mousemove', (e) => {
            tooltip.style.left = (e.clientX + 15) + 'px';
            tooltip.style.top = (e.clientY + 15) + 'px';
        });

        metric.addEventListener('mouseleave', () => {
            tooltip.style.display = 'none';
        });
    });
}

// ===== BASCULE DES DATASETS =====
function toggleDataset(datasetIndex) {
    if (!mainChart) return;

    // Récupère l'état actuel de visibilité
    const isVisible = mainChart.isDatasetVisible(datasetIndex);
    const legendItems = document.querySelectorAll('.tv-legend-item');

    if (isVisible) {
        // Masquer le dataset
        mainChart.hide(datasetIndex);
        legendItems[datasetIndex].style.opacity = '0.3';
        legendItems[datasetIndex].style.textDecoration = 'line-through';
    } else {
        // Afficher le dataset
        mainChart.show(datasetIndex);
        legendItems[datasetIndex].style.opacity = '1';
        legendItems[datasetIndex].style.textDecoration = 'none';
    }

    // Déclenche l'effet de glitch visuel
    if (typeof triggerGlitchEffect === 'function') {
        triggerGlitchEffect(200);
    }
}

// ===== INITIALISATION AU CHARGEMENT =====
document.addEventListener('DOMContentLoaded', function() {
    console.log('Initialisation des graphiques TV...');
    
    // Fonction pour attendre que les données CSV soient disponibles
    function waitForData(maxAttempts = 50) {
        return new Promise((resolve, reject) => {
            let attempts = 0;
            
            const checkData = () => {
                attempts++;
                
                if (window.CSV_DATA && window.CSV_DATA.years) {
                    console.log('Données CSV chargées:', Object.keys(window.CSV_DATA.years));
                    resolve();
                } else if (attempts >= maxAttempts) {
                    console.warn('Données CSV non chargées après 50 tentatives');
                    resolve();
                } else {
                    setTimeout(checkData, 100);
                }
            };
            
            checkData();
        });
    }
    
    // Attendre les données puis initialiser
    waitForData().then(() => {
        setTimeout(() => {
            initYears();
            initMediaFilters();
            initMainChart();
            initChartNavigation();
            
            // Initialiser les stats avec le filtre actuel
            if (typeof updateMathematicalStats === 'function') {
                const year = years[currentYearIndex] || '2019';
                updateMathematicalStats(year, currentMediaFilter);
            }
            
            console.log('Graphiques TV et filtres initialisés');
            console.log('Filtre initial:', currentMediaFilter);
            
        }, 500);
    });
    
    setTimeout(initStatsTooltips, 1000);
});



// ===== EXPORT DES FONCTIONS POUR UTILISATION GLOBALE =====
window.navigateChart = navigateChart;
window.navigateYear = navigateYear;
window.initMainChart = initMainChart;
window.updateMathematicalStats = updateMathematicalStats;
window.initMediaFilters = initMediaFilters;
window.selectAllChannels = selectAllChannels;
window.deselectAllChannels = deselectAllChannels;
window.refreshDashboard = refreshDashboard;
window.getFilteredData = getFilteredData;
window.TV_COLORS = TV_COLORS;
window.toggleDataset = toggleDataset;
window.renderEvolutionChartD3 = renderEvolutionChartD3;
window.renderBubbleChartD3 = renderBubbleChartD3;
window.renderRadialStackedChartD3 = renderRadialStackedChartD3;
window.refreshD3Chart = refreshD3Chart;